{{-- resources/views/zapatera/catalogo/index.blade.php --}}
@extends('layouts.app')

@section('content')
@php
  use App\Models\Category;

  // Estado actual
  $categorySlug = request('category_slug');
  $search       = request('search');
  $attrFilters  = (array) request('attr', []); // ej: ['color'=>'negro','talla'=>'41']

  // Categorías para el sidebar (puedes moverlo al controlador si prefieres)
  $categories = Category::orderBy('name')->get();

  // Helper para construir URL manteniendo query, cambiando/parchando pares
  function qurl(array $merge = [], array $forget = []) {
      $q = request()->query();
      foreach ($forget as $k) unset($q[$k]);
      $q = array_merge($q, $merge);
      // eliminar null/'' explícitos
      $q = array_filter($q, fn($v) => !is_null($v) && $v!=='');
      return route('catalogo', $q);
  }
@endphp

<div class="container mx-auto px-4 py-8">
  <h1 class="text-xl font-semibold text-gray-900 mb-2">Catálogo de Productos</h1>
  <p class="text-gray-600 mb-6">Filtra por categoría y atributos. Haz clic en la imagen para ver la galería.</p>

  <div class="grid grid-cols-1 md:grid-cols-4 gap-6">
    {{-- SIDEBAR --}}
    <aside class="md:col-span-1">
      <div class="nlx-card">
        <div class="nlx-card-body space-y-4">
          {{-- Categorías --}}
          <div>
            <div class="text-[11px] tracking-wide text-gray-500 uppercase mb-2">Categorías</div>

            <div class="space-y-2">
              {{-- Todas --}}
              <a href="{{ qurl(['category_slug'=>null,'page'=>null]) }}"
                 class="block w-full rounded-md px-3 py-2 text-sm font-medium
                        {{ $categorySlug ? 'bg-white border border-base-border text-gray-700 hover:bg-gray-50' : 'bg-accent text-white hover:bg-accent-600' }}">
                Todas
              </a>

              {{-- Lista de categorías --}}
              @foreach($categories as $cat)
                @php $isActive = $cat->slug === $categorySlug; @endphp
                <a href="{{ qurl(['category_slug'=>$cat->slug,'page'=>null]) }}"
                   class="block w-full rounded-md px-3 py-2 text-sm font-medium
                          {{ $isActive ? 'bg-indigo-50 text-indigo-700 border border-indigo-100' : 'bg-white border border-base-border text-gray-700 hover:bg-gray-50' }}">
                  {{ $cat->name }}
                </a>
              @endforeach
            </div>
          </div>

          {{-- Filtros dinámicos por atributos --}}
          <div class="pt-1">
            <div class="text-[11px] tracking-wide text-gray-500 uppercase mb-2">Subcategorías</div>

            <form method="GET" action="{{ route('catalogo') }}" class="space-y-3">
              {{-- Mantener todo lo que ya viene (excepto page) --}}
              @foreach(request()->except(['page','attr']) as $k=>$v)
                <input type="hidden" name="{{ $k }}" value="{{ $v }}">
              @endforeach

              {{-- Atributos (pasados por el controlador como $attributes) --}}
              @foreach(($attributes ?? []) as $attr)
                @php
                  $attrSlug  = $attr->slug;
                  $current   = $attrFilters[$attrSlug] ?? '';
                  $values    = $attr->values ?? collect();
                @endphp
                <div class="space-y-1">
                  <label class="text-xs text-gray-600">{{ $attr->name }}</label>
                  <select name="attr[{{ $attrSlug }}]"
                          class="w-full rounded-md border-base-border focus:border-accent focus:ring-accent">
                    <option value="">Todas</option>
                    @foreach($values as $val)
                      <option value="{{ $val->slug }}" @selected($current === $val->slug)>{{ $val->value }}</option>
                    @endforeach
                  </select>
                </div>
              @endforeach

              {{-- Buscar --}}
              <div class="space-y-1">
                <label class="text-xs text-gray-600">Buscar</label>
                <input type="search" name="search"
                       value="{{ $search }}"
                       placeholder="Modelo o SKU..."
                       class="w-full rounded-md border-base-border focus:border-accent focus:ring-accent">
              </div>

              <div class="flex gap-2">
                <button class="nlx-btn-accent flex-1">Filtrar</button>
                @php
                  $hasFilters = filled($categorySlug) || filled($search) || collect($attrFilters)->filter()->isNotEmpty();
                @endphp
                @if($hasFilters)
                  <a href="{{ route('catalogo') }}"
                     class="inline-flex items-center justify-center rounded-md border border-indigo-200 bg-indigo-50 px-3 py-2 text-sm font-semibold text-indigo-700 hover:bg-indigo-100">
                    Limpiar
                  </a>
                @endif
              </div>
            </form>
          </div>
        </div>
      </div>
    </aside>

    {{-- GRID productos --}}
    <section class="md:col-span-3">
      @if($products->count())
        <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 mb-4">
            @php
            // Normaliza cualquier path a una URL navegable
            $toUrl = function($raw) {
                $raw = trim($raw ?? '');
                if ($raw === '') return null;
                if (preg_match('~^https?://~i', $raw)) return $raw;                // URL absoluta
                if (str_starts_with($raw, 'public/storage/')) return asset(substr($raw,7));
                if (str_starts_with($raw, 'storage/'))        return asset($raw);  // /storage/...
                if (\Storage::disk('public')->exists($raw))   return \Storage::url($raw);
                if (file_exists(public_path($raw)))           return asset($raw);
                return null;
            };
            $placeholder = asset('img/placeholder_zapato_detalle.svg');
            @endphp

            @foreach ($products as $p)
                @php
                $isArr  = is_array($p);
                $id     = $isArr ? ($p['id'] ?? null) : $p->id;
                $name   = $isArr ? ($p['name'] ?? '') : $p->name;
                $sku    = $isArr ? ($p['sku'] ?? null) : $p->sku;

                // Precio (array => price o base_price; modelo => price o base_price)
                $priceN = $isArr
                            ? ($p['price'] ?? ($p['base_price'] ?? 0))
                            : ($p->price ?? ($p->base_price ?? 0));

                // Imagen (si vienes del controlador con 'image' absoluta, úsala tal cual)
                $image  = $isArr
                            ? ($p['image'] ?? asset('img/placeholder_zapato_detalle.svg'))
                            : (
                                // Fallbacks por si llega modelo
                                ($toUrl($p->primaryImage->url ?? $p->primaryImage->path ?? null)
                                    ?: asset('img/placeholder_zapato_detalle.svg'))
                                );

                // Stock (array => 'stock' ya calculado; modelo => suma de stocks)
                $stock  = $isArr ? ($p['stock'] ?? 0) : ($p->stocks?->sum('qty') ?? 0);

                // Pedido (WhatsApp) y galería
                $orderUrl   = $id ? route('whatsapp.order', $id) : '#';
                $galleryUrl = $id ? route('catalogo.gallery', ['product'=>$id, 'i'=>0]) : null;
                @endphp

                <x-product-card
                :title="$name"
                :sku="$sku"
                :price="'$'.number_format((float)$priceN, 2)"
                :image="$image"
                :stock="$stock"
                :url="$orderUrl"
                :gallery-url="$galleryUrl"   {{-- (opcional) ver nota abajo --}}
                />
            @endforeach


        </div>

        {{-- Paginación (usa tu componente). Fallback a links() si no existe --}}
        @isset($products)
          @if (class_exists(\Illuminate\Pagination\Paginator::class))
            @if (View::exists('components.ui.pagination'))
              <x-ui.pagination :paginator="$products" :window="2" :show-summary="false">
                <x-slot:middle>
                  <x-ui.per-page class="hidden sm:inline-flex" />
                </x-slot:middle>
              </x-ui.pagination>
            @else
              <div class="mt-4">{{ $products->links() }}</div>
            @endif
          @endif
        @endisset
      @else
        <div class="text-gray-500">No hay productos.</div>
      @endif
    </section>
  </div>
</div>

{{-- ===== Modal de Galería (mismo que en Productos admin) ===== --}}
<div
  x-data="galleryModal()"
  x-init="init()"
  x-show="isOpen"
  x-transition.opacity
  @keydown.escape.window="close()"
  @gallery:open.window="open($event.detail)"
  class="fixed inset-0 z-50 flex items-center justify-center bg-black/50"
  style="display:none;"
>
  <div class="relative w-[95vw] max-w-7xl rounded-lg bg-white shadow-xl" @click.self="close()">
    <button
      @click="close()"
      class="absolute right-2 top-2 inline-flex h-8 w-8 items-center justify-center rounded-full bg-gray-100 text-gray-600 hover:bg-gray-200"
      aria-label="Cerrar"
    >✕</button>

    <div class="p-0" x-ref="body" id="galleryModalBody"></div>
  </div>
</div>
@endsection

@push('styles')
<style>
  /* Si tu parcial trae un botón de cerrar propio, lo ocultamos en modal embed */
  #galleryModalBody .amz-close { display: none !important; }
</style>
@endpush

@push('scripts')
<script>
// Intercepta click normal para abrir modal-embed; respeta Ctrl/Meta/Shift y rueda
document.addEventListener('click', (e) => {
  const a = e.target.closest('a[aria-label="Ver galería del producto"]');
  if (!a) return;
  if (e.ctrlKey || e.metaKey || e.shiftKey || e.button === 1) return;
  e.preventDefault();
  window.dispatchEvent(new CustomEvent('gallery:open', { detail: { url: a.href } }));
});

// Componente Alpine del modal (mismo que en admin)
function galleryModal() {
  return {
    isOpen: false,
    bodyEl: null,

    init() { this.bodyEl = this.$refs.body; },

    open({ url }) {
      this.isOpen = true;
      this._renderSpinner();

      const u = new URL(url, window.location.origin);
      u.searchParams.set('embed', '1');

      fetch(u.toString(), { headers: { 'X-Requested-With': 'XMLHttpRequest' } })
        .then(res => {
          if (!res.ok) throw new Error('HTTP ' + res.status);
          return res.text();
        })
        .then(html => {
          this.bodyEl.innerHTML = html;
          this._wireGallery(this.bodyEl);
        })
        .catch(() => {
          // Fallback: si falla el fetch, navega a la página completa
          window.location.href = url;
        });
    },

    close() {
      this.isOpen = false;
      this.bodyEl.innerHTML = '';
    },

    _renderSpinner() {
      this.bodyEl.innerHTML = `
        <div class="flex min-h-[60vh] items-center justify-center p-6">
          <div class="text-center">
            <div class="mx-auto mb-2 inline-block h-6 w-6 animate-spin rounded-full border-2 border-gray-300 border-t-indigo-600"></div>
            <div class="text-sm text-gray-500">Cargando galería...</div>
          </div>
        </div>
      `;
    },

    _wireGallery(root) {
      const grid   = root.querySelector('.amz-grid');
      const main   = root.querySelector('#amz-main');
      const thumbs = root.querySelector('.amz-thumbs');
      if (!grid || !main || !thumbs) return;

      const itemNodes = thumbs.querySelectorAll('.amz-thumb img, .amz-thumb picture img, .amz-thumb source[srcset]');
      const items = [];
      itemNodes.forEach(node => {
        const src = node.getAttribute('src') || node.getAttribute('srcset') || '';
        if (src) items.push({ url: src });
      });

      let idx = 0;
      try {
        const initial = new URLSearchParams(window.location.search).get('i');
        if (initial !== null) idx = Math.max(0, Math.min(items.length - 1, parseInt(initial, 10) || 0));
      } catch(_) {}

      const paint = () => {
        if (!items.length) return;
        idx = (idx + items.length) % items.length;
        main.src = items[idx].url;
        root.querySelectorAll('.amz-thumb').forEach((t, i) => t.classList.toggle('active', i === idx));
      };

      thumbs.addEventListener('click', (e) => {
        const el = e.target.closest('.amz-thumb');
        if (!el) return;
        const imgs = Array.from(thumbs.querySelectorAll('.amz-thumb'));
        const i = imgs.indexOf(el);
        if (i >= 0) { idx = i; paint(); }
      });

      thumbs.addEventListener('mouseover', (e) => {
        const el = e.target.closest('.amz-thumb');
        if (!el) return;
        const imgs = Array.from(thumbs.querySelectorAll('.amz-thumb'));
        const i = imgs.indexOf(el);
        if (i >= 0 && i !== idx) { idx = i; paint(); }
      });

      const onKey = (e) => {
        if (!this.isOpen || !items.length) return;
        if (e.key === 'ArrowLeft')  { idx = idx - 1; paint(); }
        if (e.key === 'ArrowRight') { idx = idx + 1; paint(); }
      };
      document.addEventListener('keydown', onKey);
      this.$watch('isOpen', (open) => { if (!open) document.removeEventListener('keydown', onKey); });

      main.style.maxHeight = '60vh';
      main.style.width = '100%';
      main.style.objectFit = 'contain';

      paint();
    }
  }
}
</script>
@endpush
