<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Models\Category; // asegúrate de tener el modelo
use App\Models\Product;  // opcional (solo si prefieres Eloquent para insertar)
use App\Models\Currency;

class ProductsFromModeloSeeder extends Seeder
{
    public function run(): void
    {
        // Mapa de currencies: code/alias -> id (BD nueva)
        $currencyMap = DB::table('currencies')->pluck('id', 'code')->toArray();
        $primaryCurrencyId = Currency::where('is_primary', true)->value('id');

        // Query a las tablas viejas (ya importadas a esta misma BD)
        $q = DB::table('modelo as m')
            ->leftJoin('producto as p', 'p.id', '=', 'm.producto_id')
            ->leftJoin('moneda as mc', 'mc.id', '=', 'm.moneda')        // moneda costo
            ->leftJoin('moneda as mv', 'mv.id', '=', 'm.moneda_venta')  // moneda venta
            ->select([
                'm.id',
                'm.nombre',
                'm.code_bar',
                'm.precio',
                'm.precio_publico',
                'p.nombre as producto_nombre',     // nombre de la categoría
                'mc.alias as moneda_alias_costo',  // alias = code
                'mv.alias as moneda_alias_venta',  // alias = code
            ])
            ->orderBy('m.id');

        // Procesar en chunks
        $q->chunk(500, function ($rows) use ($currencyMap, $primaryCurrencyId) {
            foreach ($rows as $r) {
                // 1) Categoría por nombre (crear si no existe)
                $categoryId = null;
                if (!empty($r->producto_nombre)) {
                    $category = Category::where('name', $r->producto_nombre)->first();

                    if (!$category) {
                        $slug = $this->generateUniqueCategorySlug($r->producto_nombre);
                        $category = Category::create([
                            'name' => $r->producto_nombre,
                            'slug' => $slug,
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);
                    }
                    $categoryId = $category->id;
                } else {
                    // fallback si no hay nombre de categoría
                    $fallbackName = 'Sin categoría';
                    $category = Category::where('name', $fallbackName)->first();
                    if (!$category) {
                        $slug = $this->generateUniqueCategorySlug($fallbackName);
                        $category = Category::create([
                            'name' => $fallbackName,
                            'slug' => $slug,
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);
                    }
                    $categoryId = $category->id;
                }

                // 2) Monedas: alias -> currencies.id (si falla, usar principal)
                $costCurrencyId = $primaryCurrencyId;
                if (!empty($r->moneda_alias_costo)) {
                    $costCurrencyId = $currencyMap[$r->moneda_alias_costo] ?? $primaryCurrencyId;
                }

                $baseCurrencyId = $primaryCurrencyId;
                if (!empty($r->moneda_alias_venta)) {
                    $baseCurrencyId = $currencyMap[$r->moneda_alias_venta] ?? $primaryCurrencyId;
                }

                // 3) SKU único
                $sku = $this->generateUniqueSku($r->nombre, $r->id);

                // 4) Insertar product (campos solicitados + mínimos)
                DB::table('products')->insert([
                    'category_id'             => $categoryId,
                    'name'                    => $r->nombre,
                    'sku'                     => $sku,
                    'barcode'                 => $r->code_bar,
                    'cost'                    => $r->precio ?: 0,
                    'cost_currency_id'        => $costCurrencyId,
                    'base_price'              => $r->precio_publico ?: 0,
                    'base_price_currency_id'  => $baseCurrencyId,
                    'created_at'              => now(),
                    'updated_at'              => now(),
                ]);
            }
        });
    }

    /**
     * Genera un SKU único basado en el nombre y un sufijo.
     * Evita colisiones si products.sku es unique.
     */
    private function generateUniqueSku(?string $name, $fallbackId = null): string
    {
        $base = Str::slug((string) $name);
        if ($base === '') {
            $base = 'item';
        }

        $sku = $base;
        $i = 1;

        // Si ya existe, agrega sufijos -1, -2, ... o usa fallbackId
        while (
            DB::table('products')->where('sku', $sku)->exists()
        ) {
            $i++;
            $sku = $base . '-' . ($fallbackId ?? $i);
        }

        return $sku;
    }

    /**
     * Genera un slug único para categories.slug
     */
    private function generateUniqueCategorySlug(string $name): string
    {
        $base = Str::slug($name);
        if ($base === '') $base = 'categoria';

        $slug = $base;
        $i = 1;

        while (DB::table('categories')->where('slug', $slug)->exists()) {
            $i++;
            $slug = $base . '-' . $i;
        }
        return $slug;
    }

}
