<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void {
        if (!Schema::hasTable('inventory_movements')) return;

        // 1) Añadir/ajustar columnas objetivo
        Schema::table('inventory_movements', function (Blueprint $t) {
            if (!Schema::hasColumn('inventory_movements','product_id')) {
                $t->unsignedBigInteger('product_id')->nullable()->after('batch_id')->index();
            }
            if (!Schema::hasColumn('inventory_movements','attribute_values')) {
                $t->json('attribute_values')->nullable()->after('product_id');
            }
            if (!Schema::hasColumn('inventory_movements','note')) {
                $t->string('note',255)->nullable()->after('qty');
            }
            if (!Schema::hasColumn('inventory_movements','moved_at')) {
                $t->dateTime('moved_at')->nullable()->index()->after('note');
            }
            if (!Schema::hasColumn('inventory_movements','user_id')) {
                $t->unsignedBigInteger('user_id')->nullable()->after('note')->index();
            }
        });

        // 2) Unificar ref -> note si existe dato en ref
        if (Schema::hasColumn('inventory_movements','ref')) {
            DB::statement("UPDATE `inventory_movements` SET `note` = COALESCE(`note`, `ref`) WHERE `ref` IS NOT NULL AND `ref` <> ''");
        }

        // 3) Asegurar ENUM en inglés
        DB::statement("ALTER TABLE `inventory_movements`
            MODIFY COLUMN `type` ENUM('in','out','adjust') NOT NULL");

        // 4) (Opcional) Migrar variant_id -> product_id/attribute_values si hay tabla de variantes
        if (Schema::hasColumn('inventory_movements','variant_id') && Schema::hasTable('product_variants')) {
            // Asume product_variants: id, product_id, attribute_values(JSON)
            DB::statement("
                UPDATE `inventory_movements` im
                JOIN `product_variants` pv ON pv.id = im.variant_id
                SET im.product_id = COALESCE(im.product_id, pv.product_id),
                    im.attribute_values = COALESCE(im.attribute_values, pv.attribute_values)
            ");
        }

        // 5) Hacer product_id NOT NULL si lo deseas (cuando ya esté poblado)
        // Schema::table('inventory_movements', function (Blueprint $t) {
        //     $t->unsignedBigInteger('product_id')->nullable(false)->change();
        // });

        // 6) Limpiar columnas antiguas
        Schema::table('inventory_movements', function (Blueprint $t) {
            if (Schema::hasColumn('inventory_movements','ref'))       $t->dropColumn('ref');
            if (Schema::hasColumn('inventory_movements','variant_id'))$t->dropColumn('variant_id');
        });
    }

    public function down(): void {
        if (!Schema::hasTable('inventory_movements')) return;

        // re-crear columnas antiguas opcionalmente
        Schema::table('inventory_movements', function (Blueprint $t) {
            if (!Schema::hasColumn('inventory_movements','variant_id')) $t->unsignedBigInteger('variant_id')->nullable()->after('batch_id');
            if (!Schema::hasColumn('inventory_movements','ref'))        $t->string('ref',255)->nullable()->after('qty');
            // no quitamos product_id/attribute_values/moved_at/note
        });

        DB::statement("ALTER TABLE `inventory_movements`
            MODIFY COLUMN `type` ENUM('in','out','adjust') NOT NULL");
    }
};
