<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        // Evita errores por FKs en orden de borrado
        Schema::disableForeignKeyConstraints();

        // 1) Limpieza de tablas antiguas
        if (Schema::hasTable('variant_stock')) {
            // Quita FK a product_variants si existe
            try { Schema::table('variant_stock', fn (Blueprint $t) => $t->dropForeign(['variant_id'])); } catch (\Throwable $e) {}
            Schema::dropIfExists('variant_stock');
        }

        // Si existía la tabla de variantes antigua, elimínala
        Schema::dropIfExists('product_variants');

        // 2) Ajuste de tablas que dependían de variantes
        // stock_movements: remove product_variant_id y usar attribute_values JSON
        if (Schema::hasTable('stock_movements')) {
            Schema::table('stock_movements', function (Blueprint $t) {
                if (Schema::hasColumn('stock_movements','product_variant_id')) {
                    try { $t->dropForeign(['product_variant_id']); } catch (\Throwable $e) {}
                    $t->dropColumn('product_variant_id');
                }
                if (!Schema::hasColumn('stock_movements','attribute_values')) {
                    $t->json('attribute_values')->nullable()->after('product_id');
                }
            });
        }

        // 3) Crear tabla stocks desde cero (combinaciones dinámicas)
        if (!Schema::hasTable('stocks')) {
            Schema::create('stocks', function (Blueprint $table) {
                $table->id();
                $table->foreignId('product_id')->constrained()->cascadeOnDelete();
                $table->json('attribute_values')->nullable(); // { color:"negro", talla:"41", ... }
                $table->integer('qty')->default(0);
                $table->timestamps();
            });
        } else {
            // Si ya existe, la reseteamos (sin conservar datos)
            Schema::table('stocks', function (Blueprint $table) {
                // Asegura columnas mínimas
                if (!Schema::hasColumn('stocks','attribute_values')) {
                    $table->json('attribute_values')->nullable()->after('product_id');
                }
                if (!Schema::hasColumn('stocks','qty')) {
                    $table->integer('qty')->default(0)->after('attribute_values');
                }
            });
            // Vacía la tabla
            \DB::table('stocks')->truncate();
        }

        Schema::enableForeignKeyConstraints();
    }

    public function down(): void
    {
        Schema::disableForeignKeyConstraints();

        // Re-crear mínimas para volver atrás (sin datos)
        Schema::create('product_variants', function (Blueprint $table) {
            $table->id();
            $table->foreignId('product_id')->constrained()->cascadeOnDelete();
            // columnas mínimas; ajusta si quieres
            $table->timestamps();
        });

        Schema::create('variant_stock', function (Blueprint $table) {
            $table->id();
            $table->unsignedBigInteger('variant_id');
            $table->integer('qty')->default(0);
            $table->timestamps();

            $table->foreign('variant_id')->references('id')->on('product_variants')->cascadeOnDelete();
        });

        // Revertir ajuste de stock_movements
        if (Schema::hasTable('stock_movements')) {
            Schema::table('stock_movements', function (Blueprint $t) {
                if (Schema::hasColumn('stock_movements','attribute_values')) {
                    $t->dropColumn('attribute_values');
                }
                if (!Schema::hasColumn('stock_movements','product_variant_id')) {
                    $t->unsignedBigInteger('product_variant_id')->nullable()->after('product_id');
                    $t->foreign('product_variant_id')->references('id')->on('product_variants')->nullOnDelete();
                }
            });
        }

        // Puedes dropear stocks si deseas un down “estricto”
        // Schema::dropIfExists('stocks');

        Schema::enableForeignKeyConstraints();
    }
};
