<?php

namespace App\View\Components\Currency;

use Closure;
use Illuminate\Contracts\View\View;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Schema;
use Illuminate\View\Component;

class RatesBar extends Component
{
    /** @var \Illuminate\Support\Collection<int, object> */
    public Collection $currencies;

    public ?object $primary = null;
    public Collection $items;
    public string $primaryCode = '—';
    public int $decimals;
    public bool $showBoth;

    public string $variant;

    /**
     * @param Collection|array|null $currencies
     */
    public function __construct(
        mixed $currencies = null,
        int $decimals = 4,
        bool $showBoth = false,
        ?int $cacheSeconds = 300, // opcional: 5 min
        string $variant = 'scroll',
    ) {
        $this->decimals = $decimals;
        $this->showBoth = $showBoth;
        $this->variant  = $variant;

        // 1) Si vienen por prop, normaliza
        $col = $this->normalize($currencies);

        // 2) Si NO vienen, auto-cargar desde BD (y opcionalmente cachear)
        if ($col->isEmpty()) {
            $col = $this->loadFromDatabase($cacheSeconds);
        }

        $this->currencies = $col;
        $this->build();
    }

    /** Carga desde BD con fallback seguro si la tabla no existe */
    private function loadFromDatabase(?int $cacheSeconds): Collection
    {
        // Evita errores en primeras migraciones
        if (!Schema::hasTable('currencies')) {
            return collect();
        }

        // Cache opcional
        $key = 'ratesbar:active';
        if ($cacheSeconds && $cacheSeconds > 0) {
            return Cache::remember($key, $cacheSeconds, function () {
                return $this->fetchActiveCurrencies();
            });
        }

        return $this->fetchActiveCurrencies();
    }

    private function fetchActiveCurrencies(): Collection
    {
        // Usa los scopes si los agregaste; si no, deja el query explícito
        if (class_exists(\App\Models\Currency::class)) {
            return \App\Models\Currency::query()
                ->active()
                ->forRatesBar()
                ->get()
                ->map(fn($c) => (object)[
                    'code'            => strtoupper($c->code),
                    'rate_to_primary' => (float) $c->rate_to_primary,
                    'is_primary'      => (bool) $c->is_primary,
                ]);
        }
        return collect();
    }

    private function normalize(mixed $currencies): Collection
    {
        $col = $currencies instanceof Collection ? $currencies : collect($currencies ?? []);
        return $col->map(function ($c) {
            $code = strtoupper(is_array($c) ? ($c['code'] ?? '') : ($c->code ?? ''));
            $rate = (float) (is_array($c) ? ($c['rate_to_primary'] ?? 0) : ($c->rate_to_primary ?? 0));
            $isP  = (bool) (is_array($c) ? ($c['is_primary'] ?? false) : ($c->is_primary ?? false));
            return (object)[
                'code'            => $code,
                'rate_to_primary' => $rate,
                'is_primary'      => $isP,
            ];
        })->filter(fn($c) => $c->code !== '');
    }

    private function build(): void
    {
        $list = $this->currencies->values();

        $primary = $list->firstWhere('is_primary', true)
            ?: $list->firstWhere('rate_to_primary', 1.0)
            ?: $list->first();

        $this->primary     = $primary ?: null;
        $this->primaryCode = $primary?->code ?? '—';

        if (!$primary) { $this->items = collect(); return; }

        $ordered = collect([$primary])->concat(
            $list->reject(fn($c) => strtoupper($c->code) === strtoupper($primary->code))
                 ->sortBy(fn($c) => $c->code)
        );

        $this->items = $ordered->map(function ($c) use ($primary) {
            $isPrimary = strtoupper($c->code) === strtoupper($primary->code);
            $rate      = (float) $c->rate_to_primary;
            $eq        = $isPrimary ? 1.0 : ($rate > 0 ? (1.0 / $rate) : 0.0);

            return [
                'code'            => $c->code,
                'rate_to_primary' => $rate,
                'eq_per_primary'  => $eq,
                'is_primary'      => $isPrimary,
            ];
        });
    }

    public function render(): View|Closure|string
    {
        return view('components.currency.rates-bar');
    }
}
