<?php

namespace App\Services;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Schema;
use App\Models\Currency;

class CurrencyService
{
    public const CACHE_KEY = 'currencies.display.v2';
    public const CACHE_TTL = 300; // 5 min

    /**
     * Normaliza monedas para UI:
     *  - code, rate_to_primary, is_primary
     *  - primary_code, inverse (1 / rate_to_primary) para mostrar “ambas”
     *  - base primero, luego A–Z
     *  - con cache
     */
    public function listForDisplay(bool $useCache = true, ?int $ttlSeconds = null): Collection
    {
        $ttl = $ttlSeconds ?? self::CACHE_TTL;

        if ($useCache && Cache::has(self::CACHE_KEY)) {
            return Cache::get(self::CACHE_KEY);
        }

        $out = collect();

        if (Schema::hasTable('currencies')) {
            $hasIsPrimary     = Schema::hasColumn('currencies', 'is_primary');
            $hasRateToPrimary = Schema::hasColumn('currencies', 'rate_to_primary');
            $hasCode          = Schema::hasColumn('currencies', 'code');
            $hasName          = Schema::hasColumn('currencies', 'name');
            $hasRate          = Schema::hasColumn('currencies', 'rate');

            $selects = ['id'];
            if ($hasCode)          $selects[] = 'code';
            if ($hasRateToPrimary) $selects[] = 'rate_to_primary';
            if ($hasIsPrimary)     $selects[] = 'is_primary';
            if ($hasName && ! $hasCode)      $selects[] = 'name';
            if ($hasRate && ! $hasRateToPrimary) $selects[] = 'rate';

            $rows = Currency::query()->select($selects)->get();

            // Fallbacks
            if (! $hasCode && $hasName) {
                $rows->each(function ($c) { $c->code = $c->code ?? $c->name; });
            }
            if (! $hasRateToPrimary) {
                $rows->each(function ($c) use ($hasRate) {
                    if (! isset($c->rate_to_primary)) $c->rate_to_primary = $hasRate ? ($c->rate ?? 1) : 1;
                });
            }
            if (! $hasIsPrimary) {
                $primaryFound = false;
                $rows->each(function ($c) use (&$primaryFound) {
                    $isPrim = (! $primaryFound) && (float) ($c->rate_to_primary ?? 0) === 1.0;
                    $c->is_primary = $isPrim;
                    if ($isPrim) $primaryFound = true;
                });
                if (! $rows->firstWhere('is_primary', true) && $rows->count()) {
                    $rows[0]->is_primary = true;
                }
            }

            // Determinar código base
            $base = optional($rows->firstWhere('is_primary', true))->code ?? 'BASE';

            // Normalizar a array para la UI (añadimos primary_code e inverse)
            $out = $rows->map(function ($c) use ($base) {
                $rate = (float) ($c->rate_to_primary ?? 1);
                $rate = $rate <= 0 ? 1.0 : $rate; // evitar /0 o negativos
                return [
                    'id'              => $c->id,
                    'code'            => (string) ($c->code ?? ''),
                    'rate_to_primary' => $rate,
                    'is_primary'      => (bool) ($c->is_primary ?? false),
                    'primary_code'    => $base,
                    'inverse'         => 1.0 / $rate,
                ];
            });

            // Orden: base primero, luego A–Z
            $out = $out
                ->sortBy(fn ($x) => $x['is_primary'] ? 0 : 1)
                ->values()
                ->sortBy(fn ($x) => $x['is_primary'] ? '0' : strtoupper($x['code']))
                ->values();
        }

        if ($useCache) Cache::put(self::CACHE_KEY, $out, $ttl);

        return $out;
    }

    /** Invalida caché (llámalo tras crear/editar una moneda) */
    public function forgetCache(): void
    {
        Cache::forget(self::CACHE_KEY);
    }

    /** Devuelve la moneda base (o null si no hay) */
    public function primary(): ?array
    {
        return $this->listForDisplay()->firstWhere('is_primary', true);
    }

    /** Conversión básica entre monedas conocidas por code */
    public function convert(float $amount, string $fromCode, string $toCode): float
    {
        $list = $this->listForDisplay();
        $from = $list->firstWhere('code', strtoupper($fromCode));
        $to   = $list->firstWhere('code', strtoupper($toCode));
        if (! $from || ! $to) return $amount;

        // Pasamos por la base:
        // amount_in_base = amount * rate_to_primary(from)
        // amount_in_to   = amount_in_base / rate_to_primary(to)
        $base = $amount * (float) $from['rate_to_primary'];
        return $base / (float) $to['rate_to_primary'];
    }
}
