<?php

namespace App\Models;

use Illuminate\Contracts\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

class Product extends Model {
    use HasFactory;
    protected $fillable = [
                            'category_id','name','sku','description','cost','base_price','barcode',
                            'cost_currency_id','base_price_currency_id'
                          ];

    public function category(){ return $this->belongsTo(Category::class); }
    public function media(){ return $this->hasMany(Media::class); }

    public function attributes()
    {
        return $this->belongsToMany(Attribute::class, 'product_attribute_values')
                    ->with('values');
    }

    public function primaryImage(){
        return $this->hasOne(Media::class)->where('is_primary', true);
    }

    public function attributeValues()
    {
        return $this->belongsToMany(AttributeValue::class, 'product_attribute_values');
    }

    public function stocks()
    {
        return $this->hasMany(\App\Models\Stock::class);
    }

    public function movements()
    {
        // ajusta el nombre de la tabla si es 'stock_movements'
        return $this->hasMany(\App\Models\InventoryMovement::class, 'product_id');
    }

    /**
     * Scope: hace LEFT JOIN a la subquery de stock_real y añade el campo.
     *
     * Uso: Product::withStockReal()->get()
     */
    public function scopeWithStockReal(Builder $query): Builder
    {
        $sub = InventoryMovement::stockRealSubquery();

        return $query
            ->leftJoinSub($sub, 'mv', function ($j) {
                $j->on('mv.product_id', '=', 'products.id');
            })
            ->addSelect('products.*')
            ->addSelect(DB::raw('COALESCE(mv.stock_real, 0) as stock_real'));
    }

    /**
     * Accessor: si no viene unido (join), lo calcula a demanda.
     * Si ya hiciste ->withStockReal(), solo devuelve el atributo.
     */
    public function getStockRealAttribute()
    {
        // Si el atributo ya viene del SELECT (por el join), úsalo.
        if (array_key_exists('stock_real', $this->attributes)) {
            return (int) $this->attributes['stock_real'];
        }

        // Cálculo puntual (consulta rápida)
        return (int) $this->movements()
            ->selectRaw("
                SUM(
                    CASE
                        WHEN type IN ('in','entrada')  THEN qty
                        WHEN type IN ('out','salida')   THEN -qty
                        ELSE 0
                    END
                ) AS s
            ")
            ->value('s') ?? 0;
    }


    public function priceForGestor(\App\Models\User $gestor): float
    {
        // base en su moneda (asumes base_price es tu precio de lista)
        $base = (float) $this->base_price;

        // override por producto para ese gestor
        $override = \App\Models\ProductUplift::where('user_id',$gestor->id)
            ->where('product_id',$this->id)
            ->value('uplift_percent');

        // fallback al uplift por defecto del gestor
        $uplift = is_null($override) ? (float) ($gestor->default_uplift_percent ?? 0) : (float) $override;

        return $base * (1 + ($uplift/100));
    }

    public function bestImageForValues(array $selectedValueIds): ?\App\Models\Media
    {
        $selected = array_map('intval', $selectedValueIds);
        $mediaSet = $this->media()->with('attributeValues:id')->get();

        $best = null;
        $bestScore = -1;

        foreach ($mediaSet as $m) {
            $mIds = $m->attributeValues->pluck('id')->all();

            if (empty($mIds)) {
                $score = 0; // sin etiquetas
            } else {
                $allMatch = empty(array_diff($mIds, $selected));
                $score = $allMatch ? count($mIds) : -1;
            }

            if ($score > $bestScore || ($score === $bestScore && ($m->is_primary ?? false) && !optional($best)->is_primary)) {
                $best = $m;
                $bestScore = $score;
            }
        }

        return $best
            ?: $this->media()->where('is_primary', true)->first()
            ?: $this->media()->first();
    }

}
