<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class InventoryMovement extends Model
{
    use HasFactory;

    protected $table = 'inventory_movements';

    /**
     * Campos según tu tabla actual:
     * id, batch_id, product_id, attribute_values (JSON),
     * type (in|out|adjust), qty, note, user_id, moved_at,
     * created_at, updated_at
     */
    protected $fillable = [
        'batch_id',
        'product_id',
        'attribute_values',
        'type',
        'qty',
        'note',
        'user_id',
        'moved_at',
    ];

    protected $casts = [
        'attribute_values' => 'array',   // JSON -> array
        'moved_at'         => 'datetime',
    ];

    // Opcional: constantes para evitar “magic strings”
    public const TYPE_IN     = 'in';
    public const TYPE_OUT    = 'out';
    public const TYPE_ADJUST = 'adjust';

    /* ===================== Relaciones ===================== */

    public function product()
    {
        return $this->belongsTo(Product::class, 'product_id');
    }

    public function batch()
    {
        // Tabla: stock_move_batches (id)
        return $this->belongsTo(StockMoveBatch::class, 'batch_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /* ===================== Helpers / Scopes ===================== */

    // Cantidad con signo (in=+, out=-, adjust=tal cual)
    public function getSignedQtyAttribute(): int
    {
        $q = (int) ($this->qty ?? 0);
        return $this->type === self::TYPE_OUT ? -$q : $q;
    }

    public function scopeForProduct($q, int $productId)
    {
        return $q->where('product_id', $productId);
    }

    public function scopeIn($q)
    {
        return $q->where('type', self::TYPE_IN);
    }

    public function scopeOut($q)
    {
        return $q->where('type', self::TYPE_OUT);
    }

    /**
     * Subquery: stock real (in - out) agrupado por product_id.
     * Reutilizable para leftJoinSub() desde cualquier consulta.
     */
    public static function stockRealSubquery(): \Illuminate\Database\Query\Builder
    {
        $movementTable = Schema::hasTable('inventory_movements')
            ? 'inventory_movements'
            : (Schema::hasTable('stock_movements') ? 'stock_movements' : null);

        // Si no existe ninguna, devolvemos un SELECT 0 para no romper joins.
        if (!$movementTable) {
            return DB::table(DB::raw('(SELECT NULL as product_id, 0 as stock_real) as mv0'));
        }

        return DB::table($movementTable)
            ->whereNotNull('product_id')
            ->selectRaw("
                product_id,
                SUM(
                    CASE
                        WHEN type IN ('in','entrada')  THEN qty
                        WHEN type IN ('out','salida')   THEN -qty
                        ELSE 0
                    END
                ) AS stock_real
            ")
            ->groupBy('product_id');
    }
}
