<?php

namespace App\Jobs;

use App\Models\Category;
use App\Models\Product;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Collection as ECol;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class ApplyCategoryDefaultsJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public function __construct(
        public int $categoryId,
        public string $mode = 'replace',            // 'replace' o 'merge'
        public bool $onlyMissing = false,           // solo productos sin subcategorías
        public string $attributesScope = 'defaults_only' // 'defaults_only' o 'all'
    ) {}

    public function handle(): void
    {
        $category = Category::with('defaultAttributeValues.attribute')->find($this->categoryId);
        if (!$category) return;

        $defaultValueIds = $category->defaultAttributeValues->pluck('id')->all();
        $defaultAttrIds  = $category->defaultAttributeValues->pluck('attribute_id')->unique()->all();

        // Recorremos los productos por chunks
        Product::where('category_id', $category->id)
            ->orderBy('id')
            ->chunkById(500, function (ECol $chunk) use ($defaultValueIds, $defaultAttrIds) {
                foreach ($chunk as $product) {
                    $rel = $product->attributeValues(); // belongsToMany

                    if ($this->onlyMissing && $rel->count() > 0) {
                        continue; // saltar productos que ya tienen algo
                    }

                    if ($this->mode === 'replace') {
                        if ($this->attributesScope === 'defaults_only') {
                            // Remover SOLO valores de los mismos atributos de los defaults, luego adjuntar defaults
                            $current = $rel->get(['attribute_values.id','attribute_id']);
                            $toDetach = $current->filter(fn($v) => in_array($v->attribute_id, $defaultAttrIds))->pluck('id');
                            if ($toDetach->isNotEmpty()) {
                                $rel->detach($toDetach->all());
                            }
                            $rel->syncWithoutDetaching($defaultValueIds);
                        } else { // 'all'
                            // Borrar todos y dejar exactamente los defaults
                            $rel->sync($defaultValueIds);
                        }
                    } else { // merge
                        $rel->syncWithoutDetaching($defaultValueIds);
                        // (opcional) si quieres quitar valores conflictivos de los mismos atributos:
                        //  - deja lo de arriba simple para no quitar nada existente
                    }
                }
            });
    }
}
