<?php

namespace App\Http\Controllers\Zapatera;

use App\Http\Controllers\Controller;
use App\Models\Media;
use App\Models\Product;
use App\Models\ProductMedia;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Arr;

class ProductMediaController extends Controller
{
    public function storeAjax(Product $product, Request $request)
    {
        if (!$request->expectsJson() && !$request->ajax()) {
            throw ValidationException::withMessages(['upload' => 'Solicitud inválida.']);
        }

        $request->validate([
            'images'   => ['required'],
            'images.*' => ['file','mimes:jpeg,jpg,png,webp','max:5120'],
        ]);

        $files = Arr::wrap($request->file('images'));
        if (empty($files)) {
            throw ValidationException::withMessages(['images' => 'No se recibió ningún archivo.']);
        }

        $uploaded = [];

        foreach ($files as $file) {
            $path = $file->store('product_media', 'public');

            $media = new \App\Models\Media();
            $media->product_id = $product->id;
            $media->path       = $path;
            $media->save();

            // 1) Obtén la URL
            $url = Storage::disk('public')->url($media->path); // puede ser "storage/...", "/storage/..." o "http://host/tienda/storage/..."

            // 2) Quédate SOLO con el path (si era absoluta)
            $pathOnly = parse_url($url, PHP_URL_PATH) ?: $url;

            // 3) Corrige prefijo /tienda si aparece
            $pathOnly = preg_replace('#/tienda/+(storage/)#i', '/$1', $pathOnly);

            // 4) Asegura slash inicial
            if (!str_starts_with($pathOnly, '/')) {
                $pathOnly = '/'.$pathOnly;
            }

            // 5) Cache-buster
            $busted = $pathOnly.(str_contains($pathOnly,'?') ? '&' : '?').'t='.time();

            $uploaded[] = [
                'id'         => $media->id,
                'url'        => $busted,           // <-- SIEMPRE "/storage/....?t=..."
                'is_primary' => (bool)($media->is_primary ?? false),
                'chips'      => [],
            ];
        }

        return response()->json([
            'message'  => 'Imágenes subidas correctamente.',
            'uploaded' => $uploaded,
            'product'  => [
                'id'               => $product->id,
                'primary_media_id' => $product->primary_media_id,
            ],
        ], 201);
    }

    public function destroyAjax(Product $product, Media $media, Request $request)
    {
        if ($media->product_id !== $product->id) {
            abort(404);
        }

        $path = $media->path;
        $media->delete();
        if ($path && Storage::disk('public')->exists($path)) {
            Storage::disk('public')->delete($path);
        }

        return response()->json([
            'message' => "Imagen #{$media->id} eliminada.",
            'media_id' => (int)$media->id,
        ]);
    }
}
