<?php

namespace App\Http\Controllers\Zapatera;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\{Order, OrderItem, Product, ProductVariant, VariantStock, Gestor};
use App\Services\PricingService;

class OrderController extends Controller
{
    public function store(Request $request, PricingService $pricing)
    {
        $data = $request->validate([
            'token' => ['nullable','string'],
            'items' => ['required','array'],
            'items.*.variant_id' => ['required','integer','exists:product_variants,id'],
            'items.*.qty' => ['required','integer','min:1'],
        ]);

        $gestor = $data['token'] ? Gestor::where('token', $data['token'])->first() : null;
        $uplift = $request->float('p', null);

        return DB::transaction(function() use ($data, $gestor, $uplift, $pricing) {
            $order = Order::create([
                'gestor_id' => optional($gestor)->id,
                'status' => 'nuevo',
                'total' => 0,
            ]);

            $total = 0;
            foreach($data['items'] as $it){
                $variant = ProductVariant::with('product','stock')->findOrFail($it['variant_id']);
                $price = $pricing->priceFor($gestor, $variant->product, $uplift);
                $qty = (int)$it['qty'];

                // Stock check
                if (($variant->stock->qty ?? 0) < $qty) {
                    abort(422, "Stock insuficiente para variante {$variant->id}");
                }

                // Reserve stock (simple)
                $variant->stock->decrement('qty', $qty);

                OrderItem::create([
                    'order_id' => $order->id,
                    'variant_id' => $variant->id,
                    'qty' => $qty,
                    'price_unit' => $price,
                ]);

                $total += $qty * $price;
            }

            $order->update(['total' => round($total,2)]);
            return response()->json(['order_id' => $order->id, 'total' => $order->total]);
        });
    }
}
