<?php

namespace App\Http\Controllers\Zapatera;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\{Product, Category, ProductVariant, Gestor};
use App\Services\PricingService;

use Illuminate\Support\Facades\Storage;

class CatalogController extends Controller
{
    public function index(Request $request, PricingService $pricing)
    {
        $token  = $request->query('g');
        $uplift = $request->float('p', null);
        $gestor = $token ? \App\Models\Gestor::where('token', $token)->first() : null;

        $categorySlug = $request->get('category_slug');
        $search       = $request->get('search');
        $attrFilters  = (array) $request->get('attr', []); // ej: ['color'=>'negro','talla'=>'41']

        // ===== NUEVO: categorías raíz con hijos + categoría activa =====
        $categories = Category::query()
        ->with(['children' => fn($q) => $q->orderBy('name')])
        ->whereNull('parent_id')
        ->orderBy('name')
        ->get(['id','name','slug','parent_id']);

        $activeCategory = null;
        if ($categorySlug) {
            $activeCategory = Category::with('parent')->where('slug', $categorySlug)->first();
        }

        // Productos base (¡ojo!: fuera variants, dentro stocks)
        $query = \App\Models\Product::with([
            'primaryImage',
            'stocks',                    // <-- combinaciones dinámicas
            'category',
            'attributeValues.attribute', // para filtros por atributo
        ]);

        if ($categorySlug) {
            $cat = \App\Models\Category::where('slug',$categorySlug)->first();
            if ($cat) {
                $query->where('category_id',$cat->id);
            }
        }

        if ($search) {
            $query->where(function($q) use ($search){
                $q->where('name','like',"%{$search}%")
                ->orWhere('sku','like',"%{$search}%");
            });
        }

        // Filtro por subcategorías dinámicas (attrs)
        foreach ($attrFilters as $attrSlug => $valueSlug) {
            if ($valueSlug === null || $valueSlug === '') continue;

            $query->whereHas('attributeValues', function($q) use ($attrSlug, $valueSlug) {
                $q->whereHas('attribute', fn($qa) => $qa->where('slug',$attrSlug))
                ->where('slug', $valueSlug);
            });
        }

        // Ordenar por nombre de Categoria y por nombre
        $query->orderBy(
                        Category::select('name')
                            ->whereColumn('categories.id', 'products.category_id')
                        )
              ->orderBy('name');

        // Paginación + mapeo (sin N+1 para precios)
        $perPage = (int) request('per_page', 12);
        $products = $query
            ->paginate($perPage)
            ->through(function ($p) use ($gestor, $uplift, $pricing) {

                $item = [
                    'id'       => $p->id,
                    'name'     => $p->name,
                    'sku'      => $p->sku,
                    'stock'    => $p->stocks->sum('qty'),
                    'category' => optional($p->category)->name,
                ];

                // ⬇️ AQUÍ van las líneas que preguntas
                $imagePathPrimary = optional($p->primaryImage)->url
                                    ?? optional($p->primaryImage)->path
                                    ?? null;

                $item['image'] = $this->toUrl($imagePathPrimary)
                    // fallback a media primaria / primera
                    ?: $this->toUrl(optional($p->media->firstWhere('is_primary', true))->path ?? null)
                    ?: $this->toUrl(optional($p->media->first())->path ?? null)
                    // último recurso
                    ?: asset('img/placeholder_zapato_detalle.svg');

                if ($gestor || $uplift !== null) {
                    $item['price'] = $pricing->priceFor($gestor, $p, $uplift);
                }

                return $item;
            });


        // FACETAS dinámicas
        $filterableAttrs = \App\Models\Attribute::where('is_filterable', true)
            ->with('values')
            ->get();

        // Limitar valores a presentes en la categoría actual (opcional)
        if ($categorySlug) {
            $presentValues = \App\Models\AttributeValue::whereHas('products', function($q) use ($categorySlug){
                $q->whereHas('category', fn($c)=>$c->where('slug',$categorySlug));
            })->get()->groupBy('attribute_id');

            $filterableAttrs = $filterableAttrs->map(function($attr) use ($presentValues){
                $attr->setRelation('values', $presentValues->get($attr->id, collect()));
                return $attr;
            });
        }

        return view('zapatera.catalogo.index', [
            'products'        => $products,
            'gestor'          => $gestor,
            'uplift'          => $uplift,
            'attributes'      => $filterableAttrs, // para el sidebar de filtros
            'categories'      => $categories,
            'activeCategory'  => $activeCategory,
        ]);

    }

    private function toUrl(?string $path): ?string
        {
            $raw = trim((string) $path);
            if ($raw === '') return null;

            // URL absoluta
            if (preg_match('~^https?://~i', $raw)) return $raw;

            // /public/storage/... guardado
            if (str_starts_with($raw, 'public/storage/')) {
                return asset(substr($raw, 7)); // quita 'public/'
            }

            // /storage/...
            if (str_starts_with($raw, 'storage/')) {
                return asset($raw);
            }

            // archivo en disk('public')
            if (Storage::disk('public')->exists($raw)) {
                return Storage::url($raw); // => /storage/...
            }

            // archivo dentro de public/
            if (file_exists(public_path($raw))) {
                return asset($raw);
            }

            return null;
        }
}
