<?php

namespace App\Http\Controllers\Zapatera;

use App\Http\Controllers\Controller;
use App\Models\Product;
use Illuminate\Http\Request;

class BarcodePrintController extends Controller
{
    // Listado para imprimir múltiples etiquetas (con buscador)
    public function print(Request $request)
    {
        $q = trim($request->get('q', ''));
        $products = Product::query()
            ->when($q, fn($qq) => $qq->where(function ($w) use ($q) {
                $w->where('name', 'like', "%$q%")
                  ->orWhere('sku', 'like', "%$q%")
                  ->orWhere('barcode', 'like', "%$q%");
            }))
            ->whereNotNull('barcode')
            ->orderByDesc('id')
            ->paginate(48) // 48 por página, ajusta al tamaño de etiqueta
            ->withQueryString();

        return view('zapatera.admin.barcodes.print', compact('products', 'q'));
    }

    // Impresión rápida de UNA etiqueta por código libre
    public function quick(string $code)
    {
        $code = trim($code);
        abort_if($code === '', 404);

        return view('zapatera.admin.barcodes.quick', [
            'code' => $code,
        ]);
    }

    public function zplOne(string $code)
    {
        $code = trim($code);
        abort_if($code === '', 404);

        // Dimensiones en dots (203 dpi): 58mm x 40mm
        $W = 464; $H = 320;

        // Layout simple:
        // Nombre opcional (si llega por query ?name=...), barcode CODE128 centrado, texto humano abajo
        $name = trim(request('name',''));

        $zpl = "^XA"                 // start
            ."^PW{$W}"               // print width
            ."^LL{$H}"               // label length
            ."^LH0,0"                // origin
            ."^CI28"                 // UTF-8
            ."^BY2,2,56"             // módulo ancho=2, ratio=2, altura=56dots (~7mm)
            // Nombre (si hay), en dos líneas como máximo, centrado
            .($name !== '' ? "^FO20,10^FB".($W-40).",2,0,C,0^A0N,24,24^FD".$this->zplEsc($name)."^FS" : "")
            // Barcode centrado
            ."^FO20,".($name!==''?110:70) // y variable si hay nombre
            ."^FB".($W-40).",1,0,C,0"
            ."^BCN,100,Y,N,N"       // CODE128, 100 dots alto, imprime texto humano Y
            ."^FD".$this->zplEsc($code)."^FS"
            // Texto humano adicional (por si quieres reforzar con monospace)
            ."^FO20,".($name!==''?220:200)."^FB".($W-40).",1,0,C,0^A0N,26,26^FD".$this->zplEsc($code)."^FS"
            ."^PQ1"                 // cantidad 1
            ."^XZ";                 // end

        return response($zpl, 200, [
            'Content-Type' => 'application/zpl',
            'Content-Disposition' => 'attachment; filename="label_'.$code.'.zpl"',
        ]);
    }

    // Utilidad para escapar ^, ~, \ en ZPL
    private function zplEsc(string $s): string
    {
        return str_replace(['^','~','\\'], ['\^','\~','\\\\'], $s);
    }

    public function eplOne(string $code)
    {
        $code = trim($code);
        abort_if($code === '', 404);

        // EPL usa puntos también. 58mm≈464, 40mm≈320
        // b (barcode) con CODE128 (depende del modelo: "1" suele ser Code39;
        // muchos modelos EPL no soportan Code128; si no, usa Code39 (menos ideal).
        // Aquí te dejo un EPL básico con Code39 como fallback:
        $name = trim(request('name',''));
        $lines = [];
        $lines[] = "N";                      // Clear image buffer
        if ($name !== '') {
            // Texto centrado aproximado: ajusta X según largo
            $lines[] = "A20,10,0,2,1,1,N,\"".addslashes($name)."\"";
        }
        // Barcode Code39 (B1): X,Y,ROT,Type,Wide,Ratio,Height,Human,Data
        // Si tu impresora soporta Code128, cambia B1 por B8 y revisa manual.
        $lines[] = "B20,".($name!==''?100:60).",0,1,2,4,80,N,\"*{$code}*\"";
        $lines[] = "A20,".($name!==''?200:180).",0,2,1,1,N,\"{$code}\"";
        $lines[] = "P1";                     // Print 1 copy
        $epl = implode("\n", $lines)."\n";

        return response($epl)
                    ->header('Content-Type','application/vnd.eltron-epl')
                    ->header('Content-Disposition','attachment; filename="label_'.$code.'.epl"');
    }

    public function thermal(string $code)
    {
        $code = trim($code);
        $name = trim(request('name',''));
        abort_if($code === '', 404);
        return view('zapatera.admin.barcodes.thermal', compact('code','name'));
    }

}
