<?php

namespace App\Http\Controllers\Zapatera;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;

class AdminUserController extends Controller
{
    // Roles que se pueden asignar en este módulo (sin 'gestor')
    private const ALLOWED_ROLES = ['admin','manager','cliente'];

    public function __construct()
    {
        $this->middleware(['auth','verified','role:admin']);
    }

    public function index(Request $request)
    {
        $q = $request->get('q');
        $users = User::query()
            ->when($q, fn($qq)=>$qq->where(function($w) use($q){
                $w->where('name','like',"%$q%")
                  ->orWhere('email','like',"%$q%");
            }))
            ->orderBy('name')
            ->paginate(20)
            ->withQueryString();

        return view('zapatera.admin.users.index', compact('users','q'));
    }

    public function create()
    {
        $roles = Role::whereIn('name', self::ALLOWED_ROLES)->orderBy('name')->pluck('name')->all();
        return view('zapatera.admin.users.create', compact('roles'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name'                  => ['required','string','max:100'],
            'email'                 => ['required','email','max:150','unique:users,email'],
            'password'              => ['required','string','min:8','confirmed'],
            'roles'                 => ['array'],
            'roles.*'               => ['string', Rule::in(self::ALLOWED_ROLES)],
        ]);

        $user = User::create([
            'name'     => $data['name'],
            'email'    => $data['email'],
            'password' => Hash::make($data['password']),
        ]);

        $user->syncRoles($data['roles'] ?? []);

        return redirect()->route('zapatera.admin.users.index')
            ->with('status','Usuario creado');
    }

    public function edit(User $user)
    {
        $roles = Role::whereIn('name', self::ALLOWED_ROLES)->orderBy('name')->pluck('name')->all();
        return view('zapatera.admin.users.edit', compact('user','roles'));
    }

    public function update(Request $request, User $user)
    {
        $data = $request->validate([
            'name'                  => ['required','string','max:100'],
            'email'                 => ['required','email','max:150', Rule::unique('users','email')->ignore($user->id)],
            'password'              => ['nullable','string','min:8','confirmed'],
            'roles'                 => ['array'],
            'roles.*'               => ['string', Rule::in(self::ALLOWED_ROLES)],
        ]);

        $user->name  = $data['name'];
        $user->email = $data['email'];
        if (!empty($data['password'])) {
            $user->password = Hash::make($data['password']);
        }
        $user->save();

        // Seguridad: evita que te quites tu propio rol admin
        $roles = $data['roles'] ?? [];
        if ($user->id === auth()->id() && ! in_array('admin', $roles, true)) {
            return back()->withErrors('No puedes quitarte tu propio rol de administrador.');
        }
        $user->syncRoles($roles);

        return redirect()->route('zapatera.admin.users.index')
            ->with('status','Usuario actualizado');
    }

    public function destroy(User $user)
    {
        if ($user->id === auth()->id()) {
            return back()->withErrors('No puedes eliminar tu propio usuario.');
        }
        $user->delete();

        return back()->with('status','Usuario eliminado');
    }
}
