<?php
namespace App\Http\Controllers\Zapatera;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use App\Services\Settings; // tu servicio de caché
use Illuminate\Http\Request;

class AdminSettingsController extends Controller
{
    /** Listado + filtros */
    public function index(Request $request)
    {
        $q     = trim((string) $request->get('q'));
        $group = $request->get('group');

        $query = Setting::query()
            ->when($q, function ($qq) use ($q) {
                $qq->where(function ($w) use ($q) {
                    $w->where('key',   'like', "%{$q}%")
                      ->orWhere('type','like', "%{$q}%")
                      ->orWhere('group','like',"%{$q}%");
                });
            })
            ->when($group, fn($qq) => $qq->where('group', $group))
            ->orderBy('group')
            ->orderBy('key');

        $settings = $query->paginate(20)->withQueryString();
        $groups   = Setting::query()->select('group')->distinct()->pluck('group')->filter()->values();

        return view('zapatera.admin.settings.index', compact('settings','q','group','groups'));
    }

    /** Formulario de creación */
    public function create()
    {
        $setting = new Setting(['autoload' => true, 'type' => 'string']);
        return view('zapatera.admin.settings.create', compact('setting'));
    }

    /** Guardar nueva */
    public function store(Request $request, Settings $cache)
    {
        $data = $this->validateData($request, null);

        [$type, $payload] = $this->extractPayload($data);

        Setting::create([
            'key'      => $data['key'],
            'type'     => $type,
            'group'    => $data['group'] ?? null,
            'autoload' => (bool) ($data['autoload'] ?? true),
            'value'    => $payload,
        ]);

        $cache->clearCache();

        return redirect()->route('zapatera.admin.settings.index')
            ->with('status', 'Opción creada.');
    }

    /** Formulario de edición */
    public function edit(Setting $setting)
    {
        return view('zapatera.admin.settings.edit', compact('setting'));
    }

    /** Actualizar existente */
    public function update(Request $request, Setting $setting, Settings $cache)
    {
        $data = $this->validateData($request, $setting->id);

        [$type, $payload] = $this->extractPayload($data);

        $setting->update([
            'key'      => $data['key'],
            'type'     => $type,
            'group'    => $data['group'] ?? null,
            'autoload' => (bool) ($data['autoload'] ?? true),
            'value'    => $payload,
        ]);

        $cache->clearCache();

        return back()->with('status', 'Opción actualizada.');
    }

    /** Eliminar */
    public function destroy(Setting $setting, Settings $cache)
    {
        $setting->delete();
        $cache->clearCache();

        return redirect()->route('zapatera.admin.settings.index')
            ->with('status', 'Opción eliminada.');
    }

    /* ===================== Helpers ===================== */

    /** Valida inputs genéricos para creación/edición */
    private function validateData(Request $request, ?int $id): array
    {
        $unique = 'unique:settings,key';
        if ($id) $unique .= ',' . $id;

        return $request->validate([
            'key'          => ['required','string','max:255',$unique],
            'type'         => ['required','in:string,bool,int,integer,float,double,json,array'],
            'group'        => ['nullable','string','max:255'],
            'autoload'     => ['sometimes','boolean'],

            // valores según tipo
            'value_string' => ['nullable','string'],
            'value_bool'   => ['nullable','boolean'],
            'value_int'    => ['nullable','integer'],
            'value_float'  => ['nullable','numeric'],
            'value_json'   => ['nullable','json'],
        ]);
    }

    /** Transforma los inputs value_* al JSON almacenado ({"v": ...} o array) */
    private function extractPayload(array $data): array
    {
        $type = $data['type'];

        // Normaliza alias
        if ($type === 'integer') $type = 'int';
        if ($type === 'double')  $type = 'float';

        $payload = match ($type) {
            'string' => ['v' => (string)($data['value_string'] ?? '')],
            'bool'   => ['v' => (bool)  ($data['value_bool']   ?? false)],
            'int'    => ['v' => (int)   ($data['value_int']    ?? 0)],
            'float'  => ['v' => (float) ($data['value_float']  ?? 0)],
            'json','array' => json_decode($data['value_json'] ?? '[]', true) ?: [],
            default  => ['v' => (string)($data['value_string'] ?? '')],
        };

        return [$type, $payload];
    }
}
