<?php

namespace App\Http\Controllers\Zapatera;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use App\Models\{Product, Category, Media, Attribute, AttributeValue, Currency};
use App\Http\Requests\Zapatera\{ProductStoreRequest, ProductUpdateRequest};

class AdminProductController extends Controller
{
    public function index(Request $request)
    {
        $q = $request->get('q');
        $cat = $request->get('category_id');

        $perPage = (int) request('per_page', 12);
        $products = Product::with(['category','primaryImage'])
            ->when($cat, fn($q) => $q->where('category_id', $cat))
            ->when($q, function($query) use ($q){
                $query->where(function($w) use ($q){
                    $w->where('name','like',"%$q%")
                    ->orWhere('sku','like',"%$q%")
                    ->orWhere('barcode','like',"%$q%")
                    ->orWhereHas('category', fn($catQ) => $catQ->where('name','like',"%$q%"));
                });
            })
            // Orden: categoría (A–Z) y luego nombre del producto (A–Z)
            ->orderBy(
                Category::select('name')
                    ->whereColumn('categories.id', 'products.category_id')
            )
            ->orderBy('name')
            ->paginate($perPage)
            ->withQueryString();

        $categories = Category::orderBy('name')->get();

        return view('zapatera.admin.products.index', compact('products','categories','q','cat'));
    }


    public function create()
    {
        $categories = Category::orderBy('name')->get();
        $attributes = Attribute::with('values')->orderBy('name')->get();
        return view('zapatera.admin.products.create', compact('categories','attributes'));
    }

    public function store(ProductStoreRequest $request)
    {
        $data = $request->validated();
        $product = Product::create([
            'category_id' => $data['category_id'],
            'name'        => $data['name'],
            'sku'         => $data['sku'],
            'barcode'     => $data['barcode'] ?? null,
            'description' => $data['description'] ?? null,
            'cost'        => $data['cost'] ?? 0,
            'base_price'  => $data['base_price'] ?? 0,
            'cost_currency_id'       => $data['cost_currency_id'] ?? Currency::primary()->value('id'),
            'base_price_currency_id' => $data['base_price_currency_id'] ?? Currency::primary()->value('id'),
        ]);

        if (!empty($data['attribute_values'])) {
            $product->attributeValues()->sync($data['attribute_values']);
        }

        // if ($request->hasFile('images')) {
        //     $primaryIdx = (int) $request->get('primary_image_index', -1);
        //     foreach($request->file('images') as $i => $file){
        //         $path = $file->store('products','public');
        //         $isPrimary = ($i === $primaryIdx);
        //         Media::create([
        //             'product_id'=>$product->id,
        //             'path'=>'storage/'.$path,
        //             'is_primary'=>$isPrimary,
        //         ]);
        //     }
        // }

        return redirect()->route('zapatera.admin.products.index')
            ->with('status','Producto creado');
    }

    public function edit(Product $product)
    {
        $categories = Category::orderBy('name')->get();
        $attributes = Attribute::with('values')->orderBy('name')->get();
        $selectedAttributeValues = $product->attributeValues()->pluck('attribute_values.id')->all();
        $media = $product->media()->orderByDesc('is_primary')->get();

        return view('zapatera.admin.products.edit', compact(
            'product','categories','attributes','selectedAttributeValues','media'
        ));
    }

    public function update(ProductUpdateRequest $request, Product $product)
    {
        $data = $request->validated();
        $product->update([
            'category_id' => $data['category_id'],
            'name'        => $data['name'],
            'sku'         => $data['sku'],
            'barcode'     => $data['barcode'] ?? null,
            'description' => $data['description'] ?? null,
            'cost'        => $data['cost'] ?? 0,
            'base_price'  => $data['base_price'] ?? 0,
            'cost_currency_id'       => $data['cost_currency_id'] ?? Currency::primary()->value('id'),
            'base_price_currency_id' => $data['base_price_currency_id'] ?? Currency::primary()->value('id'),
        ]);

        $product->attributeValues()->sync($data['attribute_values'] ?? []);

        if ($request->hasFile('images')) {
            foreach($request->file('images') as $file){
                $path = $file->store('products','public');
                Media::create([
                    'product_id'=>$product->id,
                    'path'=>'storage/'.$path,
                    'is_primary'=>false,
                ]);
            }
        }

        if ($request->filled('primary_media_id')) {
            $product->media()->update(['is_primary'=>false]);
            $product->media()->where('id',$request->get('primary_media_id'))->update(['is_primary'=>true]);
        }

        if ($toDelete = $request->get('delete_media_ids')) {
            foreach($toDelete as $mid){
                $m = $product->media()->where('id',$mid)->first();
                if ($m) {
                    $p = str_replace('storage/','',$m->path);
                    @Storage::disk('public')->delete($p);
                    $m->delete();
                }
            }
        }

        return redirect()->route('zapatera.admin.products.edit',$product)->with('status','Producto actualizado');
    }

    public function destroy(Product $product)
    {
        foreach($product->media as $m){
            $p = str_replace('storage/','',$m->path);
            @Storage::disk('public')->delete($p);
        }
        $product->media()->delete();
        $product->attributeValues()->detach();
        $product->delete();

        return redirect()->route('zapatera.admin.products.index')->with('status','Producto eliminado');
    }

    /** Form para editar etiquetas (atributos) de una foto específica */
    public function editMedia(Product $product, Media $media)
    {
        abort_unless($media->product_id === $product->id, 404);

        $attributes = Attribute::with('values:id,attribute_id,name,slug')
            ->orderBy('name')->get();

        return view('zapatera.admin.products.media_edit', [
            'product'    => $product,
            'media'      => $media->load('attributeValues:id'),
            'attributes' => $attributes,
        ]);
    }

    /** Guarda etiquetas (attribute_value_ids) y opcionalmente is_primary */
    public function updateMedia(Request $request, Product $product, Media $media)
    {
        abort_unless($media->product_id === $product->id, 404);

        $data = $request->validate([
            'is_primary' => ['sometimes','boolean'],
            'attribute_value_ids'   => ['sometimes','array'],
            'attribute_value_ids.*' => ['integer','exists:attribute_values,id'],
        ]);

        // 1) principal (opcional)
        if ($request->has('is_primary')) {
            $media->is_primary = (bool) ($data['is_primary'] ?? false);
            $media->save();

            if ($media->is_primary) {
                Media::where('product_id', $product->id)
                    ->where('id', '!=', $media->id)
                    ->update(['is_primary' => false]);
            }
        }

        // 2) etiquetas + cache
        $media->syncAttributeValues($data['attribute_value_ids'] ?? []);

        // 3) Si pide JSON → devolvemos JSON; si no, redirect normal
        if ($request->expectsJson() || $request->ajax()) {
            // Carga fresca de relaciones para la respuesta
            $media->loadMissing('attributeValues:id,value,attribute_id');

            // chips (ya los tienes en attributes_cache) — garantizamos array
            $chips = $media->attributes_cache;
            if (!is_array($chips)) {
                $chips = json_decode($chips ?? '[]', true) ?: [];
            }

            return response()->json([
                'ok' => true,
                'message' => 'Foto actualizada y etiquetas sincronizadas.',
                'media' => [
                    'id'         => $media->id,
                    'is_primary' => (bool) $media->is_primary,
                    'chips'      => $chips,
                ],
                // devuelve cuál quedó como principal del producto
                'product' => [
                    'id'                  => $product->id,
                    'primary_media_id'    => (int) $product->media()->where('is_primary', true)->value('id') ?: null,
                ],
            ]);
        }

        // Fallback no-AJAX (por si alguien visita directo)
        return redirect()
            ->route('zapatera.admin.products.edit', $product)
            ->with('status', 'Foto actualizada y etiquetas sincronizadas.');
    }

}
