<?php

namespace App\Http\Controllers\Zapatera;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use App\Models\{
    Product,
    Category,
    Attribute,
    AttributeValue,
    Gestor,
    Stock,
    AccountingPeriod
};

class AdminPanelController extends Controller
{
    public function index()
    {
        // Métricas base
        $stats = [
            'products'   => Product::count(),
            'categories' => Category::count(),
            'gestors'    => Gestor::count(),
            'attributes' => Attribute::count(),
            'stock'      => (int) Stock::sum('qty'),
        ];

        // Ventas (si existe la tabla orders)
        $ventasTotales = 0;
        if (Schema::hasTable('orders')) {
            try {
                $ventasTotales = (float) DB::table('orders')->sum('total');
            } catch (\Throwable $e) {
                $ventasTotales = 0;
            }
        }

        // Listas
        $bajoStock = Product::withSum('stocks', 'qty')
            ->orderBy('stocks_sum_qty')
            ->having('stocks_sum_qty', '<', 5)
            ->take(5)->get();

        $topStock = Product::withSum('stocks', 'qty')
            ->orderByDesc('stocks_sum_qty')
            ->take(5)->get();

        $atributosPopulares = Attribute::withCount('values')
            ->orderByDesc('values_count')
            ->take(5)->get();

        $stockPorCategoria = Category::select('categories.name', DB::raw('SUM(stocks.qty) as total_stock'))
            ->join('products', 'products.category_id', '=', 'categories.id')
            ->join('stocks', 'stocks.product_id', '=', 'products.id')
            ->groupBy('categories.id', 'categories.name')
            ->orderByDesc('total_stock')
            ->get();

        // Últimos productos
        $latestProducts = Product::with('category')
            ->latest('id')
            ->take(8)
            ->get();

        $activePeriod = AccountingPeriod::with('local')
            ->where('activo', 1)
            ->where('cerrado', 0)
            ->latest('id')
            ->first();

        return view('zapatera.admin.dashboard', compact(
            'stats',
            'ventasTotales',
            'bajoStock',
            'topStock',
            'atributosPopulares',
            'stockPorCategoria',
            'latestProducts',
            'activePeriod'
        ));
    }
}
