<?php

namespace App\Http\Controllers\Zapatera;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use App\Models\{Product, Category, Stock, AttributeValue, Currency};
use Carbon\Carbon;
use Carbon\CarbonInterface;

class AdminInventoryController extends Controller
{

    public function index(Request $request)
    {
        // --- filtros básicos de UI
        $q   = $request->input('q', '');
        $cid = $request->input('category_id');
        $sf  = $request->input('stock', 'nonzero'); // all|zero|nonzero

        // --- subquery de stock
        $movementTable = Schema::hasTable('inventory_movements')
            ? 'inventory_movements'
            : (Schema::hasTable('stock_movements') ? 'stock_movements' : null);

        $productColumn = null;
        if ($movementTable) {
            foreach (['product_id','products_id','productId','productID','item_id'] as $col) {
                if (Schema::hasColumn($movementTable, $col)) { $productColumn = $col; break; }
            }
        }

        $stockSub = null;
        if ($movementTable && $productColumn) {
            $stockSub = DB::table("$movementTable as m")
                ->select([
                    DB::raw("m.$productColumn as product_id"),
                    DB::raw("SUM(CASE WHEN m.type IN ('out','salida') THEN -1 ELSE 1 END * COALESCE(m.qty,0)) AS stock_qty"),
                ])
                ->groupBy(DB::raw("m.$productColumn"));
        }

        // --- query principal
        $productsQ = Product::query()->from('products as p');

        if ($stockSub) {
            $productsQ->leftJoinSub($stockSub, 'st', 'st.product_id', '=', 'p.id')
                    ->addSelect(DB::raw('COALESCE(st.stock_qty,0) as stock_qty'));
        } else {
            // si no hay tabla de movimientos, expone 0
            $productsQ->addSelect(DB::raw('0 as stock_qty'));
        }

        // aplica join de monedas + selects comunes (cost/price y code/rate)
        $this->joinProductCurrencies($productsQ, 'p');

        // --- filtros
        if ($q !== '') {
            $productsQ->where(function($qq) use ($q) {
                $qq->where('p.name', 'like', "%$q%")
                ->orWhere('p.sku',  'like', "%$q%");
            });
        }

        if (!empty($cid)) {
            $productsQ->where('p.category_id', $cid);
        }

        // stock: all | zero | nonzero  (evitar HAVING)
        if ($sf === 'zero') {
            $productsQ->whereRaw('COALESCE(stock_qty,0) = 0');
        } elseif ($sf === 'nonzero') {
            $productsQ->whereRaw('COALESCE(stock_qty,0) <> 0');
        }

        $products = $productsQ
            ->orderBy('p.name')
            ->paginate(20)
            ->appends($request->query());

        // --- barra de tasas
        // $currencies = Currency::select('code','rate_to_primary','is_primary')
        //     ->where('is_active',1)
        //     ->orderByDesc('is_primary')
        //     ->orderBy('code')
        //     ->get();

        // los totales de página (para tus cards) puedes calcularlos en el Blade
        // igual que haces en history (sumando a primaria y USD usando rate_to_primary)

        return view('zapatera.admin.inventory.index', [
            'products'   => $products,
            'categories' => \App\Models\Category::select('id','name')->orderBy('name')->get(),
            // 'currencies' => $currencies,
            // si necesitas, puedes pasar también primaryCode/usdRate detectados aquí
        ]);
    }


    /** Formulario de movimientos */
    public function move(Request $request)
    {
        $categories = Category::orderBy('name')->get(['id','name']);
        $products = Product::query()
            ->leftJoin('categories as c','c.id','=','products.category_id')
            ->orderBy('c.name')
            ->orderBy('products.name')
            ->select('products.id','products.name','products.sku','products.category_id')
            ->get();
        return view('zapatera.admin.inventory.move', compact('products','categories'));
    }

    /** API: devolver TODOS los atributos del producto con sus valores */
    public function attrs(\App\Models\Product $product)
    {
        // Traer SOLO los valores asociados al producto (pivot product_attribute_values)
        // y mapear attribute_values.value -> name
        $vals = \App\Models\AttributeValue::query()
            ->select(
                'attribute_values.id',
                'attribute_values.attribute_id',
                'attribute_values.slug',
                'attribute_values.value' // <-- aquí está el "nombre" real
            )
            ->join('product_attribute_values as pav', 'pav.attribute_value_id', '=', 'attribute_values.id')
            ->where('pav.product_id', $product->id)
            ->with(['attribute:id,slug,name'])
            ->orderBy('attribute_values.attribute_id')
            ->orderBy('attribute_values.value') // <-- ordenar por value
            ->get();

        // Agrupar por atributo y formar respuesta limpia
        $grouped = $vals->groupBy('attribute_id');

        $attrs = $grouped->map(function ($list) {
            $attr = optional($list->first())->attribute;
            if (!$attr) return null;

            return [
                'id'     => $attr->id,
                'slug'   => $attr->slug,
                'name'   => $attr->name,
                'values' => $list->map(function ($v) {
                    return [
                        'id'   => $v->id,
                        'slug' => $v->slug,
                        'name' => $v->value ?: $v->slug, // <-- usar value como nombre
                    ];
                })->values(),
            ];
        })->filter()->values();

        return response()->json(['attributes' => $attrs]);
    }


    /** Guardar lote */
    public function storeMove(Request $request)
    {
        // 1) Validación de request
        $data = $request->validate([
            'type' => ['required', 'in:in,out'],            // entrada/salida
            'note' => ['nullable', 'string', 'max:255'],
            'date' => ['required', 'date'],                 // history() usa moved_at o created_at
            'rows' => ['required', 'array', 'min:1'],

            'rows.*.product_id' => ['required','integer','exists:products,id'],
            'rows.*.qty'        => ['required','integer','min:1'],

            // Desglose (acepta array o JSON)
            'rows.*.breakdown'        => ['nullable','array'],
            'rows.*.breakdown_multi'  => ['nullable','string'],
        ]);

        // 2) Detectar tabla y columna de producto (igual que en history())
        $movementTable = Schema::hasTable('inventory_movements')
            ? 'inventory_movements'
            : (Schema::hasTable('stock_movements') ? 'stock_movements' : null);

        if (!$movementTable) {
            return back()->withErrors(['general' => 'No existe tabla de movimientos.'])->withInput();
        }

        $productColumn = 'product_id';
        foreach (['product_id','products_id','productId','productID','item_id'] as $col) {
            if (Schema::hasColumn($movementTable, $col)) { $productColumn = $col; break; }
        }

        // 3) Fecha del movimiento (history() usa COALESCE(moved_at, created_at))
        $movedAt = Carbon::parse($data['date'])->setTimeFromTimeString(now()->format('H:i:s'));

        // 4) Crear batch si existe la tabla (history() agrupa por batch_id)
        $batchId = Schema::hasTable('stock_move_batches')
            ? DB::table('stock_move_batches')->insertGetId([
                'type'       => $data['type'],      // 'in' | 'out'
                'user_id'    => auth()->id(),
                'note'       => $data['note'] ?? null,
                'moved_at'   => $movedAt,
                'created_at' => now(),
                'updated_at' => now(),
            ])
            : null;

        // 5) Evitar productos duplicados en las filas
        $seen = [];
        foreach ($data['rows'] as $i => $r) {
            $pid = (int) $r['product_id'];
            if (isset($seen[$pid])) {
                return back()->withErrors([
                    "rows.$i.product_id" => 'No puedes repetir el mismo producto en el movimiento.'
                ])->withInput();
            }
            $seen[$pid] = true;
        }

        try {
            DB::transaction(function () use ($data, $movementTable, $productColumn, $batchId, $movedAt) {

                foreach ($data['rows'] as $i => $row) {
                    $productId = (int) $row['product_id'];
                    $qtyTotal  = (int) $row['qty'];
                    $note      = $data['note'] ?? null;
                    $isOut     = $data['type'] === 'out';
                    $sign      = $isOut ? -1 : +1;   // - para out, + para in

                    // Cargar producto + saber si tiene atributos
                    /** @var \App\Models\Product $product */
                    $product = \App\Models\Product::query()
                        ->with(['attributeValues:id']) // relación product->attributeValues
                        ->findOrFail($productId);

                    // ---- Parseo de desglose por attribute_value_id ----
                    // Opc A: rows[i][breakdown][<av_id>] = <qty>
                    $breakdown = [];
                    if (isset($row['breakdown']) && is_array($row['breakdown'])) {
                        foreach ($row['breakdown'] as $avid => $q) {
                            $q = (int) $q;
                            if ($q > 0) $breakdown[(int)$avid] = $q;
                        }
                    }
                    // Opc B: rows[i][breakdown_multi] = JSON
                    if (empty($breakdown) && !empty($row['breakdown_multi'])) {
                        $json = json_decode($row['breakdown_multi'], true);
                        if (is_array($json)) {
                            if (isset($json['values']) && is_array($json['values'])) {
                                foreach ($json['values'] as $avid => $q) {
                                    $q = (int) $q;
                                    if ($q > 0) $breakdown[(int)$avid] = $q;
                                }
                            } elseif (isset($json['attrs']) && is_array($json['attrs'])) {
                                foreach ($json['attrs'] as $axis) {
                                    if (isset($axis['values']) && is_array($axis['values'])) {
                                        foreach ($axis['values'] as $avid => $q) {
                                            $q = (int) $q;
                                            if ($q > 0) {
                                                $ave = (int) $avid;
                                                $breakdown[$ave] = ($breakdown[$ave] ?? 0) + $q;
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }

                    // ---- VALIDACIONES para SALIDA (out) ----
                    if ($isOut) {
                        // 1) Stock total del producto
                        $productStock = (int) (DB::table('stocks')
                            ->where('product_id', $product->id)
                            ->lockForUpdate()    // 🔒 bloqueo
                            ->value('qty') ?? 0);

                        if ($qtyTotal > $productStock) {
                            throw ValidationException::withMessages([
                                "rows.$i.qty" => "Stock insuficiente. Disponible: {$productStock}, solicitado: {$qtyTotal}."
                            ]);
                        }

                        // 2) Si el producto tiene atributos → exigir desglose
                        $productHasAttributes = $product->attributeValues->isNotEmpty();
                        if ($productHasAttributes) {
                            if (empty($breakdown)) {
                                throw ValidationException::withMessages([
                                    "rows.$i.breakdown" => "Debes especificar el desglose por atributos para este producto."
                                ]);
                            }

                            // 2.a) Sumatoria del desglose == qty total
                            $sumBreakdown = array_sum($breakdown);
                            if ($sumBreakdown !== $qtyTotal) {
                                throw ValidationException::withMessages([
                                    "rows.$i.breakdown" => "La sumatoria del desglose ({$sumBreakdown}) debe ser igual a la cantidad del producto ({$qtyTotal})."
                                ]);
                            }

                            // 2.b) Validar que cada attribute_value_id pertenezca al producto
                            $validAVIds = DB::table('product_attribute_values')
                                ->where('product_id', $product->id)
                                ->pluck('attribute_value_id')
                                ->map(fn($x)=>(int)$x)
                                ->all();

                            foreach ($breakdown as $avId => $q) {
                                if (!in_array((int)$avId, $validAVIds, true)) {
                                    throw ValidationException::withMessages([
                                        "rows.$i.breakdown" => "El atributo/valor ($avId) no pertenece al producto."
                                    ]);
                                }
                            }

                            // 2.c) Validar stock por attribute_value_id
                            $items = DB::table('product_stock_items')
                                ->where('product_id', $product->id)
                                ->whereIn('attribute_value_id', array_keys($breakdown))
                                ->lockForUpdate()    // 🔒 bloqueo por variante
                                ->get(['attribute_value_id','qty'])
                                ->keyBy('attribute_value_id');

                            foreach ($breakdown as $avId => $q) {
                                $available = (int)($items->get((int)$avId)->qty ?? 0);
                                if ($q > $available) {
                                    throw ValidationException::withMessages([
                                        "rows.$i.breakdown" => "Stock insuficiente para AV {$avId}. Disponible: {$available}, solicitado: {$q}."
                                    ]);
                                }
                            }
                        }
                    }

                    // ---- APLICACIÓN DEL MOVIMIENTO (saldo en vivo) ----
                    if (empty($breakdown)) {
                        // Sin desglose: ajustar solo el stock global del producto
                        DB::table('stocks')->updateOrInsert(
                            ['product_id' => $product->id],
                            [
                                'product_id' => $product->id,
                                'qty'        => DB::raw('GREATEST(0, COALESCE(qty,0) + ('.($sign * $qtyTotal).'))'),
                                'updated_at' => now(),
                                'created_at' => now(),
                            ]
                        );

                        // Registrar línea de movimiento
                        DB::table($movementTable)->insert([
                            'batch_id'        => $batchId,
                            $productColumn    => $product->id,         // 👈 alinea con history()
                            'type'            => $data['type'],        // 'in' | 'out'
                            'qty'             => $qtyTotal,
                            'meta'            => null,
                            'user_id'         => auth()->id(),
                            'note'            => $note,
                            'moved_at'        => $movedAt,
                            'created_at'      => now(),
                            'updated_at'      => now(),
                        ]);

                    } else {
                        // Con desglose: ajustar global + cada attribute_value_id
                        // 1) Global producto
                        DB::table('stocks')->updateOrInsert(
                            ['product_id' => $product->id],
                            [
                                'product_id' => $product->id,
                                'qty'        => DB::raw('GREATEST(0, COALESCE(qty,0) + ('.($sign * $qtyTotal).'))'),
                                'updated_at' => now(),
                                'created_at' => now(),
                            ]
                        );

                        // 2) Por variante
                        foreach ($breakdown as $avId => $q) {
                            DB::table('product_stock_items')->updateOrInsert(
                                ['product_id' => $product->id, 'attribute_value_id' => (int)$avId],
                                [
                                    'product_id'         => $product->id,
                                    'attribute_value_id' => (int)$avId,
                                    'qty'                => DB::raw('GREATEST(0, COALESCE(qty,0) + ('.($sign * (int)$q).'))'),
                                    'updated_at'         => now(),
                                    'created_at'         => now(),
                                ]
                            );
                        }

                        // 3) Registrar movimiento con meta JSON (desglose)
                        DB::table($movementTable)->insert([
                            'batch_id'        => $batchId,
                            $productColumn    => $product->id,         // 👈 alinea con history()
                            'type'            => $data['type'],
                            'qty'             => $qtyTotal,
                            'meta'            => json_encode(['breakdown_by_attribute_value_id' => $breakdown]),
                            'user_id'         => auth()->id(),
                            'note'            => $note,
                            'moved_at'        => $movedAt,
                            'created_at'      => now(),
                            'updated_at'      => now(),
                        ]);
                    }
                } // foreach rows
            }); // transaction
        } catch (ValidationException $ve) {
            throw $ve; // vuelve al form con errores
        } catch (\Throwable $e) {
            report($e);
            return back()->withErrors(['general' => 'No se pudo registrar el movimiento.'])->withInput();
        }

        return redirect()
            ->route('zapatera.admin.inventory.history')
            ->with('status', 'Movimiento registrado correctamente.');
    }


    public function storeMove_OLD(Request $request)
    {
        $data = $request->validate([
            'type' => ['required','in:in,out'],
            'note' => ['nullable','string','max:255'],
            'rows' => ['required','array','min:1'],
            'rows.*.product_id' => ['required','integer','exists:products,id'],
            'rows.*.qty'        => ['required','integer','min:1'],
            'rows.*.breakdown_multi' => ['nullable','string'], // JSON { attrs:[{slug,values:{value_slug:qty}}] }
            'date' => ['required','date','after_or_equal:today'],
        ]);

        $movedAt = Carbon::parse($request->input('date'), config('app.timezone'))
                         ->setTimeFromTimeString(now()->format('H:i:s'));

        // Evitar productos repetidos
        $seen = [];
        foreach ($data['rows'] as $i=>$r) {
            $pid = (int)$r['product_id'];
            if (isset($seen[$pid])) {
                return back()->withErrors(["rows.$i.product_id"=>"No se puede repetir el mismo producto en el movimiento."])->withInput();
            }
            $seen[$pid] = true;
        }

        $batchId = \Schema::hasTable('stock_move_batches')
            ? \DB::table('stock_move_batches')->insertGetId([
                'type'       => $data['type'],
                'user_id'    => auth()->id(),
                'note'       => $data['note'] ?? null,
                'moved_at'   => $movedAt ?? now(),
                'created_at' => now(),
                'updated_at' => now(),
            ])
            : null;

        $movementTable = Schema::hasTable('stock_movements') ? 'stock_movements'
                         : (Schema::hasTable('inventory_movements') ? 'inventory_movements' : null);

        foreach ($data['rows'] as $i=>$r) {
            $product = Product::findOrFail((int)$r['product_id']);
            $total   = (int)$r['qty'];
            $type    = $data['type'];
            $note    = $data['note'] ?? null;

            // Si no hay atributos cargados para el producto -> una sola línea
            $prodVals = $product->attributeValues()->with('attribute')->get();
            if ($prodVals->isEmpty()) {
                $this->applyMovement($product, [], $type, $total, $movementTable, $batchId, $note);
                continue;
            }

            // breakdown_multi: { attrs:[{slug, values:{ value_slug: qty, ... }}, ...] }
            $multi = json_decode($r['breakdown_multi'] ?? '{}', true) ?: [];
            if (!is_array($multi) || empty($multi['attrs'])) {
                return back()->withErrors([
                    "rows.$i.breakdown_multi" => "Este producto requiere desglose por atributos."
                ])->withInput();
            }

            // Validar sumas por atributo
            foreach ($multi['attrs'] as $aset) {
                if (!isset($aset['slug']) || !isset($aset['values']) || !is_array($aset['values'])) {
                    return back()->withErrors(["rows.$i.breakdown_multi"=>"Estructura inválida en el desglose."])->withInput();
                }
                $sum = array_sum(array_map('intval', $aset['values']));
                if ($sum !== $total) {
                    return back()->withErrors(["rows.$i.breakdown_multi"=>"En '{$aset['slug']}' la suma ($sum) debe ser igual al total ($total)."])->withInput();
                }
            }

            // Pesos por valor de cada atributo
            $weights = [];
            foreach ($multi['attrs'] as $aset) {
                $aslug = $aset['slug'];
                $weights[$aslug] = [];
                foreach ($aset['values'] as $vslug => $qv) {
                    $weights[$aslug][$vslug] = $total > 0 ? ((int)$qv) / $total : 0.0;
                }
            }

            // Conjunto real de atributos/valores del producto
            $productAttrs = $prodVals
                ->groupBy(fn($v)=>$v->attribute->slug)
                ->map(fn($g)=>$g->pluck('slug')->values())->toArray();

            // Completar con pesos uniformes si faltó algún atributo
            foreach ($productAttrs as $aslug => $vals) {
                if (!isset($weights[$aslug])) {
                    $uniform = 1.0 / max(count($vals),1);
                    $weights[$aslug] = array_fill_keys($vals, $uniform);
                } else {
                    foreach ($vals as $vslug) {
                        if (!array_key_exists($vslug, $weights[$aslug])) $weights[$aslug][$vslug] = 0.0;
                    }
                }
            }

            // Ejes para combinaciones
            $axes = [];
            foreach ($productAttrs as $aslug => $vals) {
                $axes[] = ['attr'=>$aslug, 'values'=>$vals];
            }

            // Enumerar combinaciones y calcular cantidad proporcional
            $combinations = [];
            $current = [];
            $calcQty = function($depth) use (&$calcQty,&$axes,&$current,&$combinations,$weights,$total) {
                if ($depth === count($axes)) {
                    $w = 1.0;
                    foreach ($current as $aslug=>$vslug) {
                        $w *= $weights[$aslug][$vslug] ?? 0.0;
                    }
                    $qty = $w * $total;
                    $combinations[] = ['attrs'=>$current, 'qty_float'=>$qty];
                    return;
                }
                $axis = $axes[$depth];
                foreach ($axis['values'] as $vslug) {
                    $current[$axis['attr']] = $vslug;
                    $calcQty($depth+1);
                }
                unset($current[$axis['attr']]);
            };
            $calcQty(0);

            // Redondeo y cierre
            $sumInt = 0;
            foreach ($combinations as &$c) {
                $c['qty_int'] = (int) floor($c['qty_float'] + 1e-9);
                $c['frac']    = $c['qty_float'] - $c['qty_int'];
                $sumInt += $c['qty_int'];
            }
            unset($c);
            $left = max(0, $total - $sumInt);
            usort($combinations, fn($x,$y)=>$y['frac'] <=> $x['frac']);
            for ($k=0; $k<$left && $k<count($combinations); $k++) {
                $combinations[$k]['qty_int']++;
            }

            // Aplicar
            foreach ($combinations as $c) {
                $qty = (int)$c['qty_int'];
                if ($qty <= 0) continue;
                $this->applyMovement($product, $c['attrs'], $type, $qty, $movementTable, $batchId, $note, $movedAt);
            }
        }

        return redirect()->route('zapatera.admin.inventory.move')->with('status','Movimiento registrado correctamente.');
    }

    /** Aplica el movimiento y registra línea */
    private function applyMovement(
        Product $product,
        array $attrs,
        string $type,       // 'in' | 'out' | 'adjust'
        int $qty,
        ?string $movementTable, // 'inventory_movements'
        ?int $batchId,
        ?string $note,
        ?CarbonInterface $movedAt = null
    ): void {
        $delta = $type === 'in' ? $qty : -$qty;

        // Stock por producto+atributos
        $stock = $product->stocks()
            ->when(empty($attrs), function($q){
                $q->whereNull('attribute_values')
                  ->orWhere('attribute_values','[]')
                  ->orWhere('attribute_values','{}');
            }, function($q) use ($attrs){
                $q->whereJsonContains('attribute_values', $attrs);
            })
            ->lockForUpdate()->first();

        if (!$stock) {
            $stock = new Stock();
            $stock->product_id       = $product->id;
            $stock->attribute_values = empty($attrs) ? null : $attrs;
            $stock->qty              = 0;
            $stock->save();
        }

        $stock->qty = max(0, (int)$stock->qty + $delta);
        $stock->save();

        if ($movementTable) {
            \DB::table($movementTable)->insert([
                'batch_id'         => $batchId,
                'product_id'       => $product->id,
                'attribute_values' => empty($attrs) ? null : json_encode($attrs),
                'type'             => $type,
                'qty'              => $qty,
                'user_id'          => auth()->id(),
                'note'             => $note,
                'moved_at'         => $movedAt ?? now(),
                'created_at'       => now(),
                'updated_at'       => now(),
            ]);
        }
    }


    /** Historial general (por lotes si existe la tabla de batches) y filtrado por producto */
    public function history(Request $request, ?Product $product = null)
    {
        // 0) Tabla movimientos + columna producto
        $movementTable = Schema::hasTable('inventory_movements')
            ? 'inventory_movements'
            : (Schema::hasTable('stock_movements') ? 'stock_movements' : null);

        if (!$movementTable) {
            return view('zapatera.admin.inventory.history', [
                'batches'         => null,
                'lines'           => collect(),
                'filteredProduct' => $product,
                'categories'      => collect(),
                'products'        => collect(),
            ]);
        }

        $productColumn = null;
        foreach (['product_id','products_id','productId','productID','item_id'] as $col) {
            if (Schema::hasColumn($movementTable, $col)) { $productColumn = $col; break; }
        }

        // 1) Filtros
        $from       = $request->date('from');
        $to         = $request->date('to');
        $categoryId = $request->input('category_id');
        $productIdQ = $request->input('product_id');
        $productId  = $productIdQ ?: ($product?->id);

        $typeInput = $request->input('type');
        $typeMap   = ['in'=>'in','entrada'=>'in','out'=>'out','salida'=>'out'];
        $typeNorm  = $typeInput ? ($typeMap[strtolower($typeInput)] ?? null) : null;

        // Catálogos
        $categories = Schema::hasTable('categories')
            ? Category::select('id','name')->orderBy('name')->get()
            : collect();

        $products = Schema::hasTable('products')
            ? Product::select('id','name','category_id')->orderBy('name')->get()
            : collect();

        // 2) Joins/Select para products + currencies (según tu esquema)
        $withProductAndCurrencies = function($qb, string $movAlias, string $prodCol) {
            // products
            $qb->leftJoin('products as p', "p.id", '=', "$movAlias.$prodCol");
            // currencies
            $qb->leftJoin('currencies as cc', 'cc.id', '=', 'p.cost_currency_id');
            $qb->leftJoin('currencies as cp', 'cp.id', '=', 'p.base_price_currency_id');

            // selects
            $qb->addSelect([
                DB::raw('p.id as product_id'),
                DB::raw('p.name as product_name'),
                DB::raw('p.sku as product_sku'),

                DB::raw('p.cost as cost'),
                DB::raw('p.base_price as price'),

                DB::raw('cc.code as cost_code'),
                DB::raw('cc.rate_to_primary as cost_rate'),

                DB::raw('cp.code as price_code'),
                DB::raw('cp.rate_to_primary as price_rate'),

                DB::raw('p.category_id'),
            ]);

            return $qb;
        };

        // 3) Vista por lotes si existe tabla de batches
        if (Schema::hasTable('stock_move_batches')) {
            // Batches (filtros por tipo + existencia de líneas que cumplan filtros)
            $batchesQ = DB::table('stock_move_batches as b')
                ->select('b.id','b.type','b.user_id','b.note','b.created_at','b.moved_at')
                ->orderByRaw("COALESCE(b.moved_at, b.created_at) DESC");

            if ($typeNorm) $batchesQ->where('b.type', $typeNorm);

            if ($from || $to || ($productId && $productColumn) || ($categoryId && $productColumn)) {
                $batchesQ->whereExists(function($q) use ($movementTable,$productColumn,$productId,$categoryId,$from,$to) {
                    $q->from("$movementTable as m")->whereColumn('m.batch_id','b.id');
                    if ($from) $q->whereDate(DB::raw('COALESCE(m.moved_at, m.created_at)'), '>=', $from);
                    if ($to)   $q->whereDate(DB::raw('COALESCE(m.moved_at, m.created_at)'), '<=', $to);
                    if ($productId && $productColumn) $q->where("m.$productColumn", $productId);
                    if ($categoryId && Schema::hasTable('products')) {
                        $q->whereExists(function($qq) use ($productColumn,$categoryId){
                            $qq->from('products as p2')
                            ->whereColumn('p2.id', "m.$productColumn")
                            ->where('p2.category_id', $categoryId);
                        });
                    }
                });
            }

            $batches = $batchesQ->paginate(15)->appends($request->query());

            // Líneas de esos lotes
            $ids = $batches->pluck('id')->all();

            $linesQ = DB::table("$movementTable as m")->whereIn('m.batch_id', $ids);
            if ($from) $linesQ->whereDate(DB::raw('COALESCE(m.moved_at, m.created_at)'), '>=', $from);
            if ($to)   $linesQ->whereDate(DB::raw('COALESCE(m.moved_at, m.created_at)'), '<=', $to);
            if ($productId && $productColumn) $linesQ->where("m.$productColumn", $productId);

            if ($productColumn) {
                $withProductAndCurrencies($linesQ, 'm', $productColumn);
                if ($categoryId) $linesQ->where('p.category_id', $categoryId);
            }

            $lines = $linesQ->select(
                        'm.*',
                        DB::raw("m.$productColumn as m_product_id"),

                        // Producto
                        DB::raw('p.id   as product_id'),
                        DB::raw('p.name as product_name'),

                        // Bloque monetario
                        DB::raw('p.cost as cost'),
                        DB::raw('p.base_price as price'),
                        DB::raw('cc.code as cost_code'),
                        DB::raw('cc.rate_to_primary as cost_rate'),
                        DB::raw('cp.code as price_code'),
                        DB::raw('cp.rate_to_primary as price_rate'),

                        // (opcional) para filtrar por categoría en el builder
                        DB::raw('p.category_id')
                    )
                    ->orderBy('m.id')
                    ->get()
                    ->groupBy('batch_id');

            return view('zapatera.admin.inventory.history', [
                'batches'         => $batches,
                'lines'           => $lines,
                'filteredProduct' => $productId ? Product::find($productId) : $product,
                'categories'      => $categories,
                'products'        => $products,
            ]);
        }

        // 4) Vista plana
        $q = DB::table("$movementTable as m");
        if ($from) $q->whereDate(DB::raw('COALESCE(m.moved_at, m.created_at)'), '>=', $from);
        if ($to)   $q->whereDate(DB::raw('COALESCE(m.moved_at, m.created_at)'), '<=', $to);
        if ($productId && $productColumn) $q->where("m.$productColumn", $productId);

        if (Schema::hasTable('stock_move_batches')) {
            $q->leftJoin('stock_move_batches as b','b.id','=','m.batch_id');
            if ($typeNorm) $q->where('b.type', $typeNorm);
        }

        if ($productColumn) {
            $withProductAndCurrencies($q, 'm', $productColumn);
            if ($categoryId) $q->where('p.category_id', $categoryId);
        }

        $lines = $q->select('m.*', DB::raw("m.$productColumn as m_product_id"))
                ->orderByRaw("COALESCE(m.moved_at, m.created_at) DESC")
                ->paginate(20)
                ->appends($request->query());

        return view('zapatera.admin.inventory.history', [
            'batches'         => null,
            'lines'           => $lines,
            'filteredProduct' => $productId ? Product::find($productId) : $product,
            'categories'      => $categories,
            'products'        => $products,
        ]);
    }

    /**
     * Reversa el stock para una línea concreta y elimina el movement.
     * Route: one_movement.destroy  DELETE /one/movements/destoy/{id}
     */
    public function destroyMovement(int $id)
    {
        $movementTable = 'inventory_movements';
        if (!Schema::hasTable($movementTable)) {
            return back()->withErrors('No existe la tabla de movimientos.');
        }

        $m = DB::table($movementTable)->where('id', $id)->first();
        if (!$m) {
            return back()->withErrors('Movimiento no encontrado.');
        }

        try {
            DB::transaction(function () use ($m, $movementTable) {
                $productId = (int)$m->product_id;
                $qty       = (int)$m->qty;
                $type      = (string)$m->type;

                // delta que revierte el movimiento
                $delta = ($type === 'in') ? -$qty : +$qty;

                // attribute_values puede ser string/json o null
                $attrs = null;
                if (!is_null($m->attribute_values)) {
                    $attrs = is_string($m->attribute_values)
                        ? (json_decode($m->attribute_values, true) ?: [])
                        : (array)$m->attribute_values;
                }

                // Buscar/crear stock y ajustar
                $stock = Stock::query()
                    ->when(empty($attrs), function ($q) {
                        $q->whereNull('attribute_values')
                          ->orWhere('attribute_values', '[]')
                          ->orWhere('attribute_values', '{}');
                    }, function ($q) use ($attrs) {
                        $q->whereJsonContains('attribute_values', $attrs);
                    })
                    ->where('product_id', $productId)
                    ->lockForUpdate()
                    ->first();

                if (!$stock) {
                    $stock = new Stock();
                    $stock->product_id = $productId;
                    $stock->attribute_values = empty($attrs) ? null : $attrs;
                    $stock->qty = 0;
                    $stock->save();
                }

                $stock->qty = max(0, (int)$stock->qty + $delta);
                $stock->save();

                // Eliminar la línea
                DB::table($movementTable)->where('id', $m->id)->delete();
            });

            return back()->with('status', 'Movimiento eliminado y stock revertido.');
        } catch (\Throwable $e) {
            report($e);
            return back()->withErrors('No se pudo eliminar el movimiento: '.$e->getMessage());
        }
    }

    /**
     * Reversa y elimina TODOS los movimientos de un batch para un producto.
     * Route: all_movements.destroy  DELETE /all/movements/destoy/{batch_id, producto_id}
     */
    public function destroyAllMovements(int $batch_id, int $producto_id)
    {
        $movementTable = 'inventory_movements';
        if (!Schema::hasTable($movementTable)) {
            return back()->withErrors('No existe la tabla de movimientos.');
        }

        $rows = DB::table($movementTable)
            ->where('batch_id', $batch_id)
            ->where('product_id', $producto_id)
            ->orderBy('id')
            ->get();

        if ($rows->isEmpty()) {
            return back()->withErrors('No hay movimientos para eliminar.');
        }

        try {
            DB::transaction(function () use ($rows, $movementTable) {
                foreach ($rows as $m) {
                    $productId = (int)$m->product_id;
                    $qty       = (int)$m->qty;
                    $type      = (string)$m->type;
                    $delta     = ($type === 'in') ? -$qty : +$qty;

                    $attrs = null;
                    if (!is_null($m->attribute_values)) {
                        $attrs = is_string($m->attribute_values)
                            ? (json_decode($m->attribute_values, true) ?: [])
                            : (array)$m->attribute_values;
                    }

                    $stock = Stock::query()
                        ->when(empty($attrs), function ($q) {
                            $q->whereNull('attribute_values')
                              ->orWhere('attribute_values', '[]')
                              ->orWhere('attribute_values', '{}');
                        }, function ($q) use ($attrs) {
                            $q->whereJsonContains('attribute_values', $attrs);
                        })
                        ->where('product_id', $productId)
                        ->lockForUpdate()
                        ->first();

                    if (!$stock) {
                        $stock = new Stock();
                        $stock->product_id = $productId;
                        $stock->attribute_values = empty($attrs) ? null : $attrs;
                        $stock->qty = 0;
                        $stock->save();
                    }

                    $stock->qty = max(0, (int)$stock->qty + $delta);
                    $stock->save();

                    DB::table($movementTable)->where('id', $m->id)->delete();
                }
            });

            return back()->with('status', 'Movimientos del producto eliminados y stock revertido.');
        } catch (\Throwable $e) {
            report($e);
            return back()->withErrors('No se pudo eliminar: '.$e->getMessage());
        }
    }

    /**
     * Reversa y elimina TODO el lote (todas sus líneas) y borra el batch.
     * Route: stock_move_baches.destroy  DELETE /stock_move_baches/destoy/{id}
     */
    public function destroyStockMoveBaches(int $id)
    {
        $movementTable = 'inventory_movements';
        $batchTable    = 'stock_move_batches';

        if (!Schema::hasTable($batchTable) || !Schema::hasTable($movementTable)) {
            return back()->withErrors('Faltan tablas de inventario.');
        }

        $batch = DB::table($batchTable)->where('id', $id)->first();
        if (!$batch) {
            return back()->withErrors('Lote no encontrado.');
        }

        $rows = DB::table($movementTable)->where('batch_id', $id)->orderBy('id')->get();

        try {
            DB::transaction(function () use ($rows, $movementTable, $batchTable, $id) {
                foreach ($rows as $m) {
                    $productId = (int)$m->product_id;
                    $qty       = (int)$m->qty;
                    $type      = (string)$m->type;
                    $delta     = ($type === 'in') ? -$qty : +$qty;

                    $attrs = null;
                    if (!is_null($m->attribute_values)) {
                        $attrs = is_string($m->attribute_values)
                            ? (json_decode($m->attribute_values, true) ?: [])
                            : (array)$m->attribute_values;
                    }

                    $stock = Stock::query()
                        ->when(empty($attrs), function ($q) {
                            $q->whereNull('attribute_values')
                              ->orWhere('attribute_values', '[]')
                              ->orWhere('attribute_values', '{}');
                        }, function ($q) use ($attrs) {
                            $q->whereJsonContains('attribute_values', $attrs);
                        })
                        ->where('product_id', $productId)
                        ->lockForUpdate()
                        ->first();

                    if (!$stock) {
                        $stock = new Stock();
                        $stock->product_id = $productId;
                        $stock->attribute_values = empty($attrs) ? null : $attrs;
                        $stock->qty = 0;
                        $stock->save();
                    }

                    $stock->qty = max(0, (int)$stock->qty + $delta);
                    $stock->save();

                    DB::table($movementTable)->where('id', $m->id)->delete();
                }

                // Eliminar el batch al final
                DB::table($batchTable)->where('id', $id)->delete();
            });

            return back()->with('status', 'Lote eliminado por completo y stock revertido.');
        } catch (\Throwable $e) {
            report($e);
            return back()->withErrors('No se pudo eliminar el lote: '.$e->getMessage());
        }
    }

    /**
     * Aplica los JOINs de monedas al alias de productos dado y agrega los selects
     * (id, name, cost/price + code/rate). Devuelve el mismo $qb para encadenar.
     *
     * @param \Illuminate\Database\Query\Builder|\Illuminate\Database\Eloquent\Builder $qb
     * @param string $pAlias alias de la tabla products (por defecto 'p')
     */
    private function joinProductCurrencies($qb, string $pAlias = 'p')
    {
        $qb->leftJoin('categories as cat', 'cat.id', '=', "$pAlias.category_id")
           ->leftJoin('currencies as cc', 'cc.id', '=', "$pAlias.cost_currency_id")
           ->leftJoin('currencies as cp', 'cp.id', '=', "$pAlias.base_price_currency_id")
           ->addSelect([
                    DB::raw("$pAlias.id           as product_id"),
                    DB::raw("$pAlias.name         as product_name"),
                    DB::raw("$pAlias.sku          as product_sku"),
                    DB::raw("$pAlias.category_id  as category_id"),

                    DB::raw("$pAlias.cost         as cost"),
                    DB::raw("$pAlias.base_price   as price"),

                    DB::raw("cc.code              as cost_code"),
                    DB::raw("cc.rate_to_primary   as cost_rate"),

                    DB::raw("cp.code              as price_code"),
                    DB::raw("cp.rate_to_primary   as price_rate"),

                    DB::raw("cat.name             as category_name"),
        ]);

        return $qb;
    }


}
