<?php

namespace App\Http\Controllers\Zapatera;

use App\Http\Controllers\Controller;
use App\Http\Requests\CurrencyStoreRequest;
use App\Http\Requests\CurrencyUpdateRequest;
use App\Models\Currency;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AdminCurrencyController extends Controller
{
    public function index(Request $request)
    {
        $q = $request->get('q');
        $currencies = Currency::query()
            ->when($q, fn($query)=>$query->where('name','like',"%$q%"))
            ->orderByDesc('is_primary')
            ->orderBy('code')
            ->paginate(20)          // <- asegura paginator
            ->withQueryString();    // <- conserva filtros en la paginación

        return view('zapatera.admin.currencies.index', compact('currencies','q'));
    }

    public function create()
    {
        return view('zapatera.admin.currencies.create');
    }

    public function store(CurrencyStoreRequest $request)
    {
        $data = $request->validated();
        $data['code'] = strtoupper($data['code']);
        $isPrimary = !empty($data['is_primary']);

        DB::transaction(function() use ($data, $isPrimary){
            if ($isPrimary) {
                $newRate = (float)$data['rate_to_primary']; // respecto a la antigua principal
                $data['rate_to_primary'] = 1;
                $new = Currency::create($data);

                $others = Currency::where('id','!=',$new->id)->get();
                foreach ($others as $c) {
                    $c->update([
                        'is_primary' => false,
                        'rate_to_primary' => (float)$c->rate_to_primary / ($newRate ?: 1),
                    ]);
                }
            } else {
                Currency::create($data);
            }
        });

        return redirect()->route('zapatera.admin.currencies.index')->with('status','Moneda creada');
    }

    public function edit(Currency $currency)
    {
        return view('zapatera.admin.currencies.edit', compact('currency'));
    }

    public function update(CurrencyUpdateRequest $request, Currency $currency)
    {
        $data = $request->validated();
        $data['code'] = strtoupper($data['code']);
        $isPrimary = !empty($data['is_primary']);

        DB::transaction(function() use ($data, $isPrimary, $currency) {
            if ($isPrimary && !$currency->is_primary) {
                $newRate = (float)$data['rate_to_primary'];
                $data['rate_to_primary'] = 1;
                $currency->update($data);

                $others = Currency::where('id','!=',$currency->id)->get();
                foreach ($others as $c) {
                    $c->update([
                        'is_primary' => false,
                        'rate_to_primary' => (float)$c->rate_to_primary / ($newRate ?: 1),
                    ]);
                }
            } else {
                if ($isPrimary) $data['rate_to_primary'] = 1;
                $currency->update($data);
                if ($currency->is_primary) {
                    Currency::where('id','!=',$currency->id)->update(['is_primary'=>false]);
                }
            }
        });

        return redirect()->route('zapatera.admin.currencies.index')->with('status','Moneda actualizada');
    }

    public function destroy(Currency $currency)
    {
        if ($currency->is_primary) {
            return back()->withErrors('No puedes eliminar la moneda principal.');
        }
        $currency->delete();
        return back()->with('status','Moneda eliminada');
    }
}
