<?php
namespace App\Http\Controllers\Zapatera;

use App\Http\Controllers\Controller;
use App\Models\Category;
use App\Models\AttributeValue;
use App\Jobs\ApplyCategoryDefaultsJob;
use Illuminate\Http\Request;

class AdminCategoryDefaultsController extends Controller
{
    /** Guardar los valores por defecto de la categoría (checkboxes) */
    public function store(Category $category, Request $request)
    {
        $ids = collect($request->input('attribute_values', []))
            ->map(fn($v) => (int) $v)
            ->filter()->unique()->values();

        $category->defaultAttributeValues()->sync($ids);

        return back()->with('status', 'Subcategorías por defecto guardadas.');
    }

    /**
     * Aplicar a productos de la categoría
     * Opciones:
     * - mode: replace|merge
     * - only_missing: 1/0  (solo a productos que no tengan subcategorías)
     * - attributes_scope: 'all' | 'defaults_only'
     */
    public function apply(Category $category, Request $request)
    {
        $request->validate([
            'mode'            => 'required|in:replace,merge',
            'only_missing'    => 'nullable|boolean',
            'attributes_scope'=> 'nullable|in:all,defaults_only',
        ]);

        $defaults = $category->defaultAttributeValues()->with('attribute')->get();
        if ($defaults->isEmpty()) {
            return back()->with('status', 'La categoría no tiene subcategorías por defecto definidas.');
        }

        // Despachar Job para hacerlo en background (seguro para miles de productos)
        ApplyCategoryDefaultsJob::dispatch(
            $category->id,
            $request->string('mode'),
            (bool) $request->boolean('only_missing'),
            $request->string('attributes_scope', 'defaults_only')
        );

        return back()->with('status', 'Se inició la aplicación masiva. Esto puede tardar unos minutos.');
    }
}
