<?php

namespace App\Http\Controllers\Zapatera;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\{Category, Attribute, AttributeValue, Product};

class AdminCategoryController extends Controller
{
    /** Lista (si usas resource) */
    public function index()
    {
        // $categories = Category::orderBy('name')->paginate(20);
        // return view('zapatera.admin.categories.index', compact('categories'));
        $q = trim((string) request('q'));
        $categories = \App\Models\Category::query()
            ->when($q, fn($qb) => $qb->where('name','like',"%{$q}%"))
            ->orderBy('name')
            ->paginate(12)
            ->withQueryString();

        return view('zapatera.admin.categories.index', compact('categories','q'));
    }

    /** Editar: muestra atributos+valores y los defaults ya guardados */
    public function edit(Category $category)
    {
        $attributes = Attribute::with('values')
            ->orderBy('name')
            ->get();

        // Para pintar “checked”
        $category->load('defaultAttributeValues.attribute');

        return view('zapatera.admin.categories.edit', compact('category', 'attributes'));
    }

    /**
     * Actualiza datos de la categoría y (opcional) defaults de subcategorías.
     * Puede propagar esos defaults a los productos de esta categoría.
     */
    public function update(Request $request, Category $category)
    {
        $data = $request->validate([
            'name'                => ['required','string','max:150'],
            'slug'                => ['nullable','string','max:150'],
            // IDs de attribute_values seleccionados como defaults
            'attribute_values'    => ['array'],
            'attribute_values.*'  => ['integer','exists:attribute_values,id'],
            // Si llega 'propagate' (on), aplicamos a los productos
            'propagate'           => ['nullable','boolean'],
        ]);

        // Actualiza campos básicos de la categoría (si procede)
        $category->fill([
            'name' => $data['name'],
            'slug' => $data['slug'] ?? $category->slug,
        ])->save();

        // Sincroniza los defaults de la categoría
        $valueIds = collect($data['attribute_values'] ?? [])->unique()->values();

        DB::transaction(function () use ($category, $valueIds, $data) {

            // 1) Guardar defaults
            $category->defaultAttributeValues()->sync($valueIds);

            // 2) ¿Propagar a productos?
            $shouldPropagate = (bool)($data['propagate'] ?? false);
            if (! $shouldPropagate || $valueIds->isEmpty()) {
                return;
            }

            // Atributos involucrados (para solo resetear esos)
            $attrIds = AttributeValue::whereIn('id', $valueIds)
                        ->pluck('attribute_id')
                        ->unique()
                        ->values();

            // Todos los productos de esta categoría
            $category->loadMissing('products.attributeValues');
            $products = $category->products; // asume hasMany Product en Category

            foreach ($products as $product) {
                // Detach de los valores que pertenezcan a los atributos afectados
                if ($attrIds->isNotEmpty()) {
                    $toDetach = $product->attributeValues()
                        ->whereIn('attribute_id', $attrIds)
                        ->pluck('attribute_values.id');
                    if ($toDetach->isNotEmpty()) {
                        $product->attributeValues()->detach($toDetach);
                    }
                }

                // Attach de los defaults nuevos (evita duplicados)
                if ($valueIds->isNotEmpty()) {
                    $product->attributeValues()->syncWithoutDetaching($valueIds);
                }
            }
        });

        // Flash de progreso
        $n = $category->products()->count();
        return redirect()
            ->route('zapatera.admin.categories.edit', $category)
            ->with('status', "Categoría actualizada. Atributos por defecto guardados"
                . ($request->boolean('propagate') ? " y propagados a {$n} productos." : '.'));
    }

    /** Crear/Guardar (si usas todo el CRUD) */
    public function create()
    {
        return view('zapatera.admin.categories.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => ['required','string','max:150'],
            'slug' => ['nullable','string','max:150'],
        ]);

        $category = Category::create($data);

        return redirect()
            ->route('zapatera.admin.categories.edit', $category)
            ->with('status', 'Categoría creada correctamente.');
    }

    public function destroy(Category $category)
    {
        $name = $category->name;
        $category->delete();

        return redirect()
            ->route('zapatera.admin.categories.index')
            ->with('status', "Categoría «{$name}» eliminada.");
    }
}
