<?php

namespace App\Http\Controllers\Zapatera;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Models\Attribute;
use App\Http\Requests\Zapatera\{AttributeStoreRequest, AttributeUpdateRequest};

class AdminAttributeController extends Controller
{
    public function index(Request $request)
    {
        $q    = trim((string)$request->input('q'));
        $sort = $request->input('sort', 'name');   // id | name | slug | is_filterable
        $dir  = $request->input('dir',  'asc');    // asc | desc

        // columna permitida (evitar SQL injection en orderBy)
        $allowedSort = ['id','name','slug','is_filterable'];
        if (!in_array($sort, $allowedSort, true)) $sort = 'name';
        if (!in_array(strtolower($dir), ['asc','desc'], true)) $dir = 'asc';

        $attributes = \App\Models\Attribute::query()
            ->when($q !== '', function ($qb) use ($q) {
                $qq = "%{$q}%";
                $qb->where(function ($w) use ($qq) {
                    $w->where('name', 'like', $qq)
                    ->orWhere('slug', 'like', $qq);
                });
            })
            ->orderBy($sort, $dir)
            ->paginate(12)
            ->withQueryString(); // 👈 mantiene ?q&sort&dir en la paginación

        return view('zapatera.admin.attributes.index', compact('attributes', 'q', 'sort', 'dir'));
    }

    public function create()
    {
        return view('zapatera.admin.attributes.create');
    }

    public function store(AttributeStoreRequest $request)
    {
        $data = $request->validated();
        if (empty($data['slug'])) {
            $data['slug'] = Str::slug($data['name']);
        }
        Attribute::create($data);

        return redirect()->route('zapatera.admin.attributes.index')
            ->with('status','Atributo creado');
    }

    public function edit(Attribute $attribute)
    {
        $values = $attribute->values()->orderBy('value')->paginate(20);
        return view('zapatera.admin.attributes.edit', compact('attribute','values'));
    }

    public function update(AttributeUpdateRequest $request, Attribute $attribute)
    {
        $data = $request->validated();
        if (empty($data['slug'])) {
            $data['slug'] = Str::slug($data['name']);
        }
        $attribute->update($data);

        return redirect()->route('zapatera.admin.attributes.edit', $attribute)
            ->with('status','Atributo actualizado');
    }

    public function destroy(Attribute $attribute)
    {
        if ($attribute->values()->exists()) {
            return back()->with('status','No se puede eliminar: el atributo tiene valores.');
        }
        $attribute->delete();
        return redirect()->route('zapatera.admin.attributes.index')->with('status','Atributo eliminado');
    }
}
