<?php

// app/Http/Controllers/Zapatera/AdminAccountingPeriodController.php
namespace App\Http\Controllers\Zapatera;

use App\Http\Controllers\Controller;
use App\Models\AccountingPeriod;
use App\Models\Local;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AdminAccountingPeriodController extends Controller
{
    public function index(Request $request)
    {
        $q         = trim((string) request('q'));
        $localId   = request('local_id');
        $estado    = request('estado', 'all');

        $perPage = (int) request('per_page', 12);
        $periods = \App\Models\AccountingPeriod::with('local')
            ->when($q, fn($qb)=> $qb->where('descripcion','like',"%{$q}%"))
            ->when(filled($localId), fn($qb)=> $qb->where('local_id', $localId))
            ->when($estado !== 'all', function ($qb) use ($estado) {
                return match ($estado) {
                    'activos'   => $qb->where('activo', 1),
                    'inactivos' => $qb->where('activo', 0),
                    'cerrados'  => $qb->where('cerrado', 1),
                    default     => $qb,
                };
            })
            ->orderByDesc('id')
            ->paginate($perPage)
            ->withQueryString();

        $locals = Local::query()
                ->orderBy('alias') // o name
                ->pluck('alias','id'); // ['id' => 'Alias']

        return view('zapatera.admin.accounting_period.index', compact('periods', 'locals','q','localId', 'estado'));;
    }

    public function create()
    {
        $locals = Local::query()
                    ->orderBy('alias') // o name
                    ->pluck('alias','id'); // ['id' => 'Alias']

        return view('zapatera.admin.accounting_period.create', compact('locals'));
    }

    public function store(Request $request)
    {
        $data = $this->validateData($request);

        $this->assertNoOverlap($data);
        if (!empty($data['activo'])) $this->ensureSingleActive($data);

        AccountingPeriod::create($data);

        return redirect()->route('zapatera.admin.accounting_period.index')
            ->with('status', 'Periodo creado correctamente.');
    }

    public function edit(AccountingPeriod $periodo)
    {
        $locals = Local::query()
                    ->orderBy('alias') // o name
                    ->pluck('alias','id'); // ['id' => 'Alias']

        return view('zapatera.admin.accounting_period.edit', compact('periodo', 'locals'));
    }

    public function update(Request $request, AccountingPeriod $periodo)
    {
        $data = $this->validateData($request);

        $this->assertNoOverlap($data, $periodo->id);
        if (!empty($data['activo'])) $this->ensureSingleActive($data, $periodo->id);

        $periodo->update($data);

        return back()->with('status', 'Periodo actualizado.');
    }

    public function destroy(AccountingPeriod $periodo)
    {
        $periodo->delete();
        return back()->with('status','Periodo eliminado.');
    }

    public function activar(AccountingPeriod $periodo)
    {
        if ($periodo->cerrado) return back()->withErrors('No se puede activar un periodo cerrado.');

        $data = $periodo->only(['local_id','fecha_inicial','fecha_final']) + ['activo'=>1];
        $this->ensureSingleActive($data, $periodo->id);
        $this->assertNoOverlap($data, $periodo->id);

        DB::transaction(function() use ($periodo) {
            AccountingPeriod::where('local_id', $periodo->local_id)->update(['activo'=>false]);
            $periodo->update(['activo'=>true]);
        });

        return back()->with('status','Periodo activado.');
    }

    public function cerrar(AccountingPeriod $periodo)
    {
        $periodo->update(['cerrado'=>true, 'activo'=>false]);
        return back()->with('status','Periodo cerrado.');
    }

    /* ================= Helpers ================= */

    protected function validateData(Request $request): array
    {
        return $request->validate([
            'local_id'      => ['required','integer'],
            'fecha_inicial' => ['required','date'],
            'fecha_final'   => ['required','date','after_or_equal:fecha_inicial'],
            'descripcion'   => ['nullable','string','max:200'],
            'cerrado'       => ['sometimes','boolean'],
            'activo'        => ['sometimes','boolean'],
        ]);
    }

    protected function assertNoOverlap(array $data, ?int $excludeId = null): void
    {
        $overlap = AccountingPeriod::query()
            ->where('local_id', $data['local_id'])
            ->when($excludeId, fn($qb)=> $qb->where('id','!=',$excludeId))
            // solape: A.start <= B.end AND A.end >= B.start
            ->whereDate('fecha_inicial','<=',$data['fecha_final'])
            ->whereDate('fecha_final','>=',$data['fecha_inicial'])
            ->exists();

        if ($overlap) {
            back()->withErrors('Las fechas se solapan con otro periodo del mismo local.')->throwResponse();
        }
    }

    protected function ensureSingleActive(array $data, ?int $excludeId = null): void
    {
        $existsActive = AccountingPeriod::query()
            ->where('local_id', $data['local_id'])
            ->when($excludeId, fn($qb)=> $qb->where('id','!=',$excludeId))
            ->where('activo', true)
            ->where('cerrado', false)
            ->exists();

        if ($existsActive) {
            back()->withErrors('Ya existe un periodo activo para este local. Desactívalo o ciérralo antes.')->throwResponse();
        }
    }
}
