<?php

namespace App\Http\Controllers\Gestor;

use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductUplift;
use Illuminate\Http\Request;

class UpliftController extends Controller
{
    public function editGlobal(Request $request)
    {
        $user = $request->user();
        return view('gestor.uplift.global', compact('user'));
    }

    public function updateGlobal(Request $request)
    {
        $data = $request->validate([
            'default_uplift_percent' => ['nullable','numeric','min:0','max:1000'],
        ]);

        $request->user()->update([
            'default_uplift_percent' => $data['default_uplift_percent'] ?? null
        ]);

        return back()->with('status','Uplift global actualizado.');
    }

    public function indexProducts(Request $request)
    {
        $user = $request->user();
        $q = $request->get('q');

        $products = Product::with('category')
            ->when($q, fn($qq)=>$qq->where(function($w) use ($q){
                $w->where('name','like',"%$q%")
                  ->orWhere('sku','like',"%$q%");
            }))
            ->orderBy('name')
            ->paginate(20)
            ->withQueryString();

        $uplifts = ProductUplift::where('user_id',$user->id)
                   ->pluck('uplift_percent','product_id');

        return view('gestor.uplift.products', compact('products','uplifts','q','user'));
    }

    public function storeOrUpdate(Request $request, Product $product)
    {
        $data = $request->validate([
            'uplift_percent' => ['required','numeric','min:0','max:1000'],
        ]);

        ProductUplift::updateOrCreate(
            ['user_id'=>$request->user()->id,'product_id'=>$product->id],
            ['uplift_percent'=>$data['uplift_percent']]
        );

        return back()->with('status','Uplift guardado para el producto.');
    }

    public function destroy(Request $request, Product $product)
    {
        ProductUplift::where('user_id',$request->user()->id)
            ->where('product_id',$product->id)
            ->delete();

        return back()->with('status','Uplift del producto eliminado (usa el global).');
    }
}
